<?php

declare(strict_types=1);

namespace Drupal\graphql_core_schema\Event;

use Drupal\Component\EventDispatcher\Event;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\TypedData\DataDefinitionInterface;

/**
 * An event that is triggered to alter entity field schema data.
 */
class AlterEntityFieldEvent extends Event {

  /**
   * Event fired to alter schema data.
   *
   * @var string
   */
  const EVENT_NAME = 'graphql_core_schema.alter_entity_field';

  public function __construct(
    protected string $gqlFieldMachineName,
    protected string $gqlFieldSchemaType,
    protected string $gqlFieldName,
    protected EntityTypeInterface $entityType,
    protected array $schemaConfiguration,
    protected ?DataDefinitionInterface $dataDefinition = NULL,
    protected ?array $fieldDefinition = NULL,
    protected ?string $gqlFieldDescription = NULL,
  ) {}

  /**
   * Get the GraphQL field machine name.
   *
   * @return string
   *   The GraphQL field machine name.
   */
  public function getGqlFieldMachineName(): ?string {
    return $this->gqlFieldMachineName;
  }

  /**
   * Set the GraphQL field machine name.
   *
   * @param string $machineName
   *   The GraphQL field machine name.
   */
  public function setGqlFieldMachineName(string $machineName): void {
    $this->gqlFieldMachineName = $machineName;
  }

  /**
   * Get the GraphQL field schema type.
   *
   * @return string
   *   The GraphQL field schema type.
   */
  public function getGqlFieldSchemaType(): ?string {
    return $this->gqlFieldSchemaType;
  }

  /**
   * Set the GraphQL schema type.
   *
   * @param string $schemaType
   *   The GraphQL schema type.
   */
  public function setGqlFieldSchemaType(string $schemaType): void {
    $this->gqlFieldSchemaType = $schemaType;
  }

  /**
   * Get the GraphQL field name.
   *
   * @return string
   *   The GraphQL field name.
   */
  public function getGqlFieldName(): ?string {
    return $this->gqlFieldName;
  }

  /**
   * Set the GraphQL field name.
   *
   * @param string $fieldName
   *   The GraphQL field name.
   */
  public function setGqlFieldFieldName(string $fieldName): void {
    $this->gqlFieldName = $fieldName;
  }

  /**
   * Get the GraphQL field description.
   *
   * @return string
   *   The GraphQL field description.
   */
  public function getGqlFieldDescription(): ?string {
    return $this->gqlFieldDescription;
  }

  /**
   * Set the GraphQL field description.
   *
   * @param string $fieldDescription
   *   The GraphQL field description.
   */
  public function setGqlFieldDescription(?string $fieldDescription = NULL): void {
    $this->gqlFieldDescription = $fieldDescription;
  }

  /**
   * Get the entity type.
   *
   * @return \Drupal\Core\Entity\EntityTypeInterface
   *   The config entity type.
   */
  public function getEntityType(): EntityTypeInterface {
    return $this->entityType;
  }

  /**
   * Get the drupal field definition array.
   *
   * @return array
   *   The drupal field definition.
   */
  public function getFieldDefinition(): ?array {
    return $this->fieldDefinition;
  }

  /**
   * Get the drupal data definition.
   *
   * @return \Drupal\Core\TypedData\DataDefinitionInterface
   *   The drupal data definition.
   */
  public function getDataDefinition(): ?DataDefinitionInterface {
    return $this->dataDefinition;
  }

  /**
   * Get the schema configuration.
   *
   * @return array
   *   The schema configuration.
   */
  public function getSchemaConfiguration(): array {
    return $this->schemaConfiguration;
  }

}
