<?php

namespace Drupal\graphql_core_schema\GraphQL;

use Drupal\Component\Plugin\ConfigurableInterface;
use Drupal\graphql\Entity\ServerInterface;
use Drupal\graphql\GraphQL\Validator;
use GraphQL\Type\Definition\TypeWithFields;

/**
 * Core composable schema aware validator.
 *
 * Because fields generated by EntitySchemaBuilder are resolved using the
 * default field resolver they appear as missing in the validation view.
 * We override the getMissingResolvers method to remove fields that are
 * resolved by the default resolver.
 */
class CoreComposableValidator extends Validator {

  /**
   * {@inheritdoc}
   */
  public function getMissingResolvers(ServerInterface $server, array $ignore_types = []) : array {
    $missing = parent::getMissingResolvers($server, $ignore_types);
    $schemaName = $server->get('schema');

    if ($schemaName !== 'core_composable') {
      return $missing;
    }

    /** @var \Drupal\graphql\Plugin\SchemaPluginInterface $plugin */
    $plugin = $this->pluginManager->createInstance($schemaName);
    if ($plugin instanceof ConfigurableInterface && $config = $server->get('schema_configuration')) {
      $plugin->setConfiguration($config[$schemaName] ?? []);
    }

    $schema = $plugin->getSchema($plugin->getResolverRegistry());

    foreach (array_keys($missing) as $typeName) {
      $type = $schema->getType($typeName);

      if ($type instanceof TypeWithFields) {
        // Filter out fields that contain the "{field" string in the
        // description. This is used by the default resolver to know which
        // property to resolve.
        $missing[$typeName] = array_filter($missing[$typeName], function ($fieldName) use ($type) {
          $field = $type->getField($fieldName);
          $description = $field->description ?? '';
          return !str_contains($description, '{field');
        });
        if (empty($missing[$typeName])) {
          unset($missing[$typeName]);
        }
      }
    }

    return $missing;
  }

}
