<?php

declare(strict_types=1);

namespace Drupal\graphql_export;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\graphql\Entity\Server;
use Drupal\graphql\Entity\ServerInterface;
use Drupal\graphql\GraphQL\Utility\Introspection;
use Drupal\graphql\Plugin\SchemaPluginManager;
use GraphQL\Utils\SchemaPrinter;

/**
 * GraphQL Export Service.
 */
class GraphqlExportService {

  /**
   * Construct a new export controller.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\graphql\GraphQL\Utility\Introspection $introspection
   *   The introspection service.
   * @param \Drupal\graphql\Plugin\SchemaPluginManager $pluginManager
   *   The schema plugin manager.
   */
  public function __construct(
    protected MessengerInterface $messenger,
    protected Introspection $introspection,
    protected SchemaPluginManager $pluginManager,
  ) {}

  /**
   * Get the schema for a server.
   *
   * @param \Drupal\graphql\Entity\ServerInterface $graphql_server
   *   The server.
   *
   * @return string
   *   The schema.
   */
  public function printSchema(ServerInterface $graphql_server): string {
    $plugin = $this->pluginManager->getInstanceFromServer($graphql_server);

    $schema = $plugin->getSchema($plugin->getResolverRegistry());
    $printed = SchemaPrinter::doPrint($schema);

    return $printed;
  }

  /**
   * Print the schema as introspection JSON.
   *
   * @param \Drupal\graphql\Entity\ServerInterface $graphql_server
   *   The server.
   *
   * @return string|null
   *   The introspection JSON.
   */
  public function printJson(ServerInterface $graphql_server): ?string {
    if ($graphql_server instanceof Server) {
      $graphql_server->setQueryDepth(0);
      $graphql_server->setQueryComplexity(0);
      $graphql_server->setDisableIntrospection(FALSE);
    }
    $introspection = $this->introspection->introspect($graphql_server);
    $printed = json_encode($introspection, JSON_THROW_ON_ERROR | JSON_PRETTY_PRINT | JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_AMP | JSON_HEX_QUOT);

    return $printed ?: NULL;
  }

}
