<?php

namespace Drupal\graphql_focal_point\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\crop\Entity\Crop;
use Drupal\focal_point\FocalPointManagerInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\image\Plugin\Field\FieldType\ImageItem;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Data producer for image focal point.
 *
 * @DataProducer(
 *   id = "focal_point",
 *   name = @Translation("Focal Point"),
 *   description = @Translation("Provides the focal point coordinates of an image field item."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Focal Point")
 *   ),
 *   consumes = {
 *     "image" = @ContextDefinition("any",
 *       label = @Translation("Image field item")
 *     )
 *   }
 * )
 */
class FocalPoint extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The focal point manager.
   *
   * @var \Drupal\focal_point\FocalPointManagerInterface
   */
  protected $focalPointManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, FocalPointManagerInterface $focal_point_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->focalPointManager = $focal_point_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('focal_point.manager')
    );
  }

  /**
   * Resolver for field focal point.
   *
   * @param \Drupal\image\Plugin\Field\FieldType\ImageItem $image_item
   *   The image field item.
   *
   * @return array|null
   *   Focal point data with the following structure:
   *   - x: int (0-100) - X coordinate as percentage
   *   - y: int (0-100) - Y coordinate as percentage  
   *   - coordinates: string - Formatted as "x,y" (e.g., "50,75")
   *   - xNormalized: float (0-1) - X coordinate normalized
   *   - yNormalized: float (0-1) - Y coordinate normalized
   *   - coordinatesNormalized: string - Formatted as "x,y" with normalized values (e.g., "0.5,0.75")
   *   Returns NULL if no focal point is available.
   */
  public function resolve(ImageItem $image_item) {
    if (!$image_item) {
      return NULL;
    }

    // Get the file entity
    $file = $image_item->entity;
    if (!$file) {
      return NULL;
    }

    // Get the crop type from focal point config
    $crop_type = \Drupal::config('focal_point.settings')->get('crop_type') ?: 'focal_point';
    
    // Find the crop entity for this file
    $crop = Crop::findCrop($file->getFileUri(), $crop_type);
    if (!$crop) {
      // No focal point set, return default center point
      return [
        'x' => 50,
        'y' => 50,
        'coordinates' => '50,50',
        'xNormalized' => 0.5,
        'yNormalized' => 0.5,
        'coordinatesNormalized' => '0.5,0.5',
      ];
    }

    // Get image dimensions
    $width = $image_item->width;
    $height = $image_item->height;
    
    if (!$width || !$height) {
      return NULL;
    }

    // Convert absolute crop coordinates to relative percentages
    $relative_coordinates = $this->focalPointManager->absoluteToRelative(
      $crop->x->value,
      $crop->y->value,
      $width,
      $height
    );

    $x_percentage = (int) $relative_coordinates['x'];
    $y_percentage = (int) $relative_coordinates['y'];

    $x_normalized = (float) ($x_percentage / 100);
    $y_normalized = (float) ($y_percentage / 100);

    return [
      'x' => $x_percentage,
      'y' => $y_percentage,
      'coordinates' => $x_percentage . ',' . $y_percentage,
      'xNormalized' => $x_normalized,
      'yNormalized' => $y_normalized,
      'coordinatesNormalized' => $x_normalized . ',' . $y_normalized,
    ];
  }

}
