<?php

namespace Drupal\graphql_search_api_query\Plugin\GraphQL\DataProducer;

use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\search_api\Entity\Index;
use Drupal\search_api\Query\QueryInterface;
use GraphQL\Error\UserError;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * @DataProducer(
 *   id = "search_api_query",
 *   name = @Translation("Search API Query"),
 *   description = @Translation("Performs a search on a Search API index."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Search results")
 *   ),
 *   consumes = {
 *     "index_id" = @ContextDefinition("string",
 *       label = @Translation("Index ID")
 *     ),
 *     "keys" = @ContextDefinition("string",
 *       label = @Translation("Search keys"),
 *       required = FALSE
 *     ),
 *     "filter" = @ContextDefinition("any",
 *       label = @Translation("Filter"),
 *       required = FALSE
 *     ),
 *     "facets" = @ContextDefinition("any",
 *       label = @Translation("Facets"),
 *       required = FALSE
 *     ),
 *     "sort" = @ContextDefinition("any",
 *       label = @Translation("Sort"),
 *       required = FALSE
 *     ),
 *     "limit" = @ContextDefinition("integer",
 *       label = @Translation("Limit"),
 *       required = FALSE
 *     ),
 *     "offset" = @ContextDefinition("integer",
 *       label = @Translation("Offset"),
 *       required = FALSE
 *     ),
 *     "language" = @ContextDefinition("string",
 *       label = @Translation("Language"),
 *       required = FALSE
 *     )
 *   }
 * )
 */
class SearchApiQuery extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  const OPERATOR_MAPPING = [
    'BETWEEN' => 'BETWEEN',
    'CONTAINS' => 'CONTAINS',
    'ENDS_WITH' => 'ENDS_WITH',
    'EQUAL' => '=',
    'GREATER_THAN' => '>',
    'GREATER_THAN_OR_EQUAL' => '>=',
    'IN' => 'IN',
    'IS_NOT_NULL' => 'IS NOT NULL',
    'IS_NULL' => 'IS NULL',
    'LIKE' => 'LIKE',
    'NOT_BETWEEN' => 'NOT BETWEEN',
    'NOT_EQUAL' => '!=',
    'NOT_IN' => 'NOT IN',
    'NOT_LIKE' => 'NOT LIKE',
    'REGEXP' => 'REGEXP',
    'SMALLER_THAN' => '<',
    'SMALLER_THAN_OR_EQUAL' => '<=',
    'STARTS_WITH' => 'STARTS_WITH',
  ];

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('language_manager')
    );
  }

  /**
   * Constructor.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LanguageManagerInterface $language_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->languageManager = $language_manager;
  }

  /**
   * Performs a search api query and returns the results.
   *
   * @param string $index_id
   *   The Search API index ID.
   * @param string $keys
   *   The search keywords.
   * @param array|null $filter
   *   Addtional filters.
   * @param array|null $facets
   *   The facets to group & filter results by. Defaults to no facets.
   * @param array|null $sort
   *   Object, containing field and direction properties for sorting.
   * @param int $limit
   *   The maximum number of results to return.
   * @param int $offset
   *   The starting offset in the result set.
   * @param string|null $language
   *   The language code to filter results by. Defaults to current language.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The field context.
   *
   * @return array
   *   The search results.
   */
  public function resolve($index_id, $keys = '', $filter = [], $facets = [], $sort = [], $limit = 50, $offset = 0, $language = NULL, FieldContext $field) {

    // Init query. 
    $index = Index::load($index_id);
    if (!$index) {
      return [
        'results' => [],
        'count' => 0,
        'facets' => [],
      ];
    }
    $query = $index->query();

    // Set keys for fulltext search.
    if (!empty($keys)) {
      $query->setProcessingLevel(QueryInterface::PROCESSING_FULL);
      $query->keys($keys);
      $parse_mode = \Drupal::service('plugin.manager.search_api.parse_mode')
        ->createInstance('direct');
      $query->setParseMode($parse_mode);
    }

    // Define facets.
    if (!empty($facets)) {
      $facetDefinitions = [];
      foreach ($facets as $facet) {
        $facetDefinitions[$facet] = [
          'field' => $facet,
          'limit' => 10,
          'operator' => 'and',
          'min_count' => 1,
          'missing' => FALSE,
        ];
      }
      $query->setOption('search_api_facets', $facetDefinitions);
    }

    // Define filters. 
    if (!empty($filter)) {
      $filterConditions = $this->buildFilterConditions($query, $filter);
      if (count($filterConditions->getConditions())) {
        $query->addConditionGroup($filterConditions);
      }
    }

    // Define sorting. 
    if (empty($sort)) {
      $query->sort('search_api_relevance', 'DESC');
    } 
    else {
      foreach ($sort as $item) {
        $direction = !empty($item['direction']) ? $item['direction'] : 'DESC';
        $query->sort($item['field'], $direction, $language);
      }
    }

    // Set range.
    $query->range($offset, $limit);

    // Set language.
    if (!$language) {
      $language = $this->languageManager->getCurrentLanguage()->getId();
    }
    $language_field = $this->getLanguageField($index);
    if ($language_field) {
      $query->addCondition($language_field, $language);
    }

    // Run query.
    $results = $query->execute();
    $resultItems = array_values($results->getResultItems());

    // Create output.
    $resultOutput = array_map(function($item) use ($language) {
      $entity = $item->getOriginalObject()->getEntity();
      if ($entity->isTranslatable() && $entity->hasTranslation($language)) {
        $entity = $entity->getTranslation($language);
      }
      return $entity;
    }, $resultItems);

    // Create facet output.
    $resultFacets = $results->getExtraData('search_api_facets');
    $resultFacetsOutput = [];
    if ($resultFacets) {
      foreach ($resultFacets as $facetField => $facetFilters) {
        foreach ($facetFilters as $facetFilter) {
          $filterValue = trim($facetFilter['filter'], '"');
          $filteredItems = array_values(array_filter($resultItems, function($item) use ($filterValue, $facetField) {
            return in_array($filterValue, $item->getField($facetField)->getValues());
          }));

          $filterValueLabel = $this->resolveFilterLabel($index, $facetField, $filterValue, $language);

          $resultFacetsOutput[] = [
            'field' => $facetField,
            'filter' => [
              'id' => $filterValue,
              'label' => $filterValueLabel,
            ],
            'count' => (int) $facetFilter['count'],
            'results' => array_map(function($item) use ($language) {
              $entity = $item->getOriginalObject()->getEntity();
              if ($entity->isTranslatable() && $entity->hasTranslation($language)) {
                $entity = $entity->getTranslation($language);
              }
              return $entity;
            }, $filteredItems),
          ];
        }
      }
    }

    return [
      'results' => $resultOutput,
      'count' => $results->getResultCount(),
      'facets' => $resultFacetsOutput,
    ];
  }
  
  /**
   * Determines the language field to use for filtering.
   *
   * @param \Drupal\search_api\Entity\Index $index
   *   The search index.
   *
   * @return string|null
   *   The language field name, or NULL if none available.
   */
  protected function getLanguageField($index) {
    // Standard Search API language field
    if ($index->getField('langcode')) {
      return 'langcode';
    }
    
    // Common alternative language field names
    foreach (['language', 'lang', 'language_code'] as $field_name) {
      if ($index->getField($field_name)) {
        return $field_name;
      }
    }
    
    return NULL;
  }

  /**
   * Recursively builds the filter condition groups.
   *
   * @param \Drupal\search_api\Query\QueryInterface $query
   *   The search query object.
   * @param array $filter
   *   The filter definitions from the field arguments.
   *
   * @return \Drupal\search_api\Query\ConditionGroup
   *   The generated condition group according to the given filter definitions.
   *
   * @throws \GraphQL\Error\Error
   *   If the given operator and value for a filter are invalid.
   */
  protected function buildFilterConditions(QueryInterface $query, array $filter) {
    $conjunction = !empty($filter['conjunction']) ? $filter['conjunction'] : 'AND';
    $group = $query->createConditionGroup($conjunction);

    // Apply filter conditions.
    $conditions = !empty($filter['conditions']) ? $filter['conditions'] : [];
    foreach ($conditions as $condition) {
      // Check if we need to disable this condition.
      if (isset($condition['enabled']) && empty($condition['enabled'])) {
        continue;
      }

      $field = $condition['field'];
      $value = !empty($condition['value']) ? $condition['value'] : NULL;
      $operator = !empty($condition['operator']) ? $condition['operator'] : NULL;

      // Map GraphQL operator to proper ConditionInterface operator.
      if (isset(static::OPERATOR_MAPPING[$operator])) {
        $operator = static::OPERATOR_MAPPING[$operator];
      }

      // We need at least a value or an operator.
      if (empty($operator) && empty($value)) {
        throw new UserError(sprintf("Missing value and operator in filter for '%s'.", $field));
      }
      // Unary operators need a single value.
      elseif (!empty($operator) && $this->isUnaryOperator($operator)) {
        if (empty($value) || count($value) > 1) {
          throw new UserError(sprintf("Unary operators must be associated with a single value (field '%s').", $field));
        }

        // Pick the first item from the values.
        $value = reset($value);
      }
      // Range operators need exactly two values.
      elseif (!empty($operator) && $this->isRangeOperator($operator)) {
        if (empty($value) || count($value) !== 2) {
          throw new UserError(sprintf("Range operators must require exactly two values (field '%s').", $field));
        }
      }
      // Null operators can't have a value set.
      elseif (!empty($operator) && $this->isNullOperator($operator)) {
        if (!empty($value)) {
          throw new UserError(sprintf("Null operators must not be associated with a filter value (field '%s').", $field));
        }
      }

      // If no operator is set, however, we default to EQUALS or IN, depending
      // on whether the given value is an array with one or more than one items.
      if (empty($operator)) {
        $value = count($value) === 1 ? reset($value) : $value;
        $operator = is_array($value) ? 'IN' : '=';
      }

      // Add the condition for the current field.
      $group->addCondition($field, $value, $operator);
    }

    // Apply nested filter group conditions.
    $groups = !empty($filter['groups']) ? $filter['groups'] : [];
    foreach ($groups as $args) {
      // By default, we use AND condition groups.
      // Conditions can be disabled. Check we are not adding an empty condition group.
      $filterConditions = $this->buildFilterConditions($query, $args);
      if (count($filterConditions->conditions())) {
        $group->condition($filterConditions);
      }
    }

    return $group;
  }

  /**
   * Checks if an operator is a unary operator.
   *
   * @param string $operator
   *   The query operator to check against.
   *
   * @return bool
   *   TRUE if the given operator is unary, FALSE otherwise.
   */
  protected function isUnaryOperator($operator) {
    $unary = ["=", "<>", "<", "<=", ">", ">=", "LIKE", "NOT LIKE", "CONTAINS", "STARTS_WITH", "ENDS_WITH"];
    return in_array($operator, $unary);
  }

  /**
   * Checks if an operator is a null operator.
   *
   * @param string $operator
   *   The query operator to check against.
   *
   * @return bool
   *   TRUE if the given operator is a null operator, FALSE otherwise.
   */
  protected function isNullOperator($operator) {
    $null = ["IS NULL", "IS NOT NULL"];
    return in_array($operator, $null);
  }

  /**
   * Checks if an operator is a range operator.
   *
   * @param string $operator
   *   The query operator to check against.
   *
   * @return bool
   *   TRUE if the given operator is a range operator, FALSE otherwise.
   */
  protected function isRangeOperator($operator) {
    $null = ["BETWEEN", "NOT BETWEEN"];
    return in_array($operator, $null);
  }

  /**
   * Resolves the label for a filter value based on the field configuration.
   *
   * @param \Drupal\search_api\Entity\Index $index
   *   The search index.
   * @param string $field_name
   *   The field name.
   * @param string $filter_value
   *   The filter value (ID).
   * @param string $language
   *   The language code.
   *
   * @return string
   *   The resolved label or the original value if no label can be determined.
   */
  protected function resolveFilterLabel($index, $field_name, $filter_value, $language) {
    // Get the field configuration from the index.
    $fields = $index->getFields();
    if (!isset($fields[$field_name])) {
      return $filter_value;
    }

    $field = $fields[$field_name];
    $property_path = $field->getPropertyPath();
    $datasource_id = $field->getDatasourceId();

    // Handle entity bundle field (type).
    if ($property_path === 'type' && strpos($datasource_id, 'entity:') === 0) {
      $entity_type_id = substr($datasource_id, 7); // Remove 'entity:' prefix
      $bundle_info = \Drupal::service('entity_type.bundle.info')->getBundleInfo($entity_type_id);
      if (isset($bundle_info[$filter_value]['label'])) {
        return (string) $bundle_info[$filter_value]['label'];
      }
      return $filter_value;
    }

    // Get the source entity type from datasource.
    if (strpos($datasource_id, 'entity:') !== 0) {
      return $filter_value; // Not an entity datasource.
    }

    $source_entity_type_id = substr($datasource_id, 7); // Remove 'entity:' prefix
    
    try {
      // Get the entity type definition.
      $entity_type_manager = \Drupal::entityTypeManager();
      $source_entity_type = $entity_type_manager->getDefinition($source_entity_type_id);
      
      // Get field definitions for this entity type.
      $field_definitions = \Drupal::service('entity_field.manager')
        ->getBaseFieldDefinitions($source_entity_type_id);
      
      // Also get bundle field definitions if this entity has bundles.
      if ($source_entity_type->hasKey('bundle')) {
        $bundles = \Drupal::service('entity_type.bundle.info')->getBundleInfo($source_entity_type_id);
        foreach (array_keys($bundles) as $bundle) {
          $bundle_fields = \Drupal::service('entity_field.manager')
            ->getFieldDefinitions($source_entity_type_id, $bundle);
          $field_definitions = array_merge($field_definitions, $bundle_fields);
        }
      }

      // Find the field definition that matches our property path.
      $target_field_definition = $this->getFieldDefinitionFromPropertyPath($field_definitions, $property_path);
      
      if (!$target_field_definition) {
        return $filter_value;
      }

      // Get the target entity type from the field definition.
      $target_entity_type_id = $this->getTargetEntityTypeFromField($target_field_definition);
      
      if (!$target_entity_type_id) {
        return $filter_value; // Not a reference field.
      }

      // Load the target entity and get its label.
      $target_entity = $entity_type_manager->getStorage($target_entity_type_id)->load($filter_value);
      if ($target_entity) {
        // Handle translations.
        if ($target_entity->isTranslatable() && $target_entity->hasTranslation($language)) {
          $target_entity = $target_entity->getTranslation($language);
        }
        
        // Return the entity label.
        return $target_entity->label();
      }
    }
    catch (\Exception $e) {
      // Log the error but don't break the response.
      \Drupal::logger('graphql_search_api_query')->warning('Error resolving filter label: @message', ['@message' => $e->getMessage()]);
    }

    // Default: return the original value if no label resolution is possible.
    return $filter_value;
  }

  /**
   * Gets the field definition from a property path.
   *
   * @param array $field_definitions
   *   Array of field definitions.
   * @param string $property_path
   *   The property path (e.g., 'field_tags', 'field_tags:entity:name').
   *
   * @return \Drupal\Core\Field\FieldDefinitionInterface|null
   *   The field definition or NULL if not found.
   */
  protected function getFieldDefinitionFromPropertyPath($field_definitions, $property_path) {
    // Handle simple field names and complex property paths.
    $field_name = explode(':', $property_path)[0];
    
    return isset($field_definitions[$field_name]) ? $field_definitions[$field_name] : NULL;
  }

  /**
   * Gets the target entity type ID from a field definition.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   *
   * @return string|null
   *   The target entity type ID or NULL if not a reference field.
   */
  protected function getTargetEntityTypeFromField($field_definition) {
    $field_type = $field_definition->getType();
    
    // Handle entity reference fields.
    if (in_array($field_type, ['entity_reference', 'entity_reference_revisions'])) {
      $settings = $field_definition->getSettings();
      return isset($settings['target_type']) ? $settings['target_type'] : NULL;
    }

    // Handle file and image fields (they reference file entities).
    if (in_array($field_type, ['file', 'image'])) {
      return 'file';
    }

    return NULL;
  }
}