# GraphQL Search API Query Extension

This module provides GraphQL integration for Drupal's Search API, allowing you to perform search queries through GraphQL with the same functionality as the core `graphql_core_schema` entity query but operating on Search API indexes instead of directly on entities.

## Features

- **Search API GraphQL Integration**: Exposes Search API queries via GraphQL data producers
- **Entity Query Compatibility**: Accepts the same arguments as the `drupal/graphql_core_schema` entity query
- **Search Index Support**: Performs queries on configurable Search API indexes
- **Faceted Search**: Built-in support for facets with configurable options
- **Dynamic Label Resolution**: Automatically resolves human-readable labels for entity reference fields

## Requirements

- Drupal 11+
- GraphQL module (`graphql:graphql`)
- Search API module (`search_api:search_api`)

## Installation

1. Enable the module:
   ```bash
   drush en graphql_search_api_query
   ```

2. Configure your Search API indexes to include any fields you want to query or facet on.

## Usage

### Basic Search Query

The module extends the GraphQL Query type with a `searchApiQuery` field that accepts the following parameters:

- `indexId` (required): The machine name of the Search API index
- `keys`: Search keywords for full-text search
- `filter`: Complex filter conditions (same structure as entity query filters)
- `facets`: Array of field names to generate facets for
- `sort`: Sorting options with field and direction
- `limit`: Maximum number of results (default: 50)
- `offset`: Starting offset for pagination
- `language`: Language code to filter results by

### Example GraphQL Query

```graphql
query SearchContent {
  searchApiQuery(
    indexId: "content_index"
    keys: "sustainability"
    filter: {
      conditions: [
        {
          field: "type"
          operator: IN
          value: ["article", "page"]
        },
        {
          field: "status"
          operator: EQUAL
          value: ["1"]
        }
      ]
      conjunction: AND
    }
    facets: ["type", "field_category"]
    sort: [
      {
        field: "search_api_relevance"
        direction: DESC
      },
      {
        field: "created"
        direction: DESC
      }
    ]
    limit: 10
    offset: 0
    language: "en"
  ) {
    results {
      ... on NodeInterface {
        id
        title
        status
        created
        entityBundle
        entityUrl {
          path
        }
      }
    }
    count
    facets {
      field
      label
      filter {
        id
        label
      }
      count
      results {
        ... on NodeInterface {
          id
          title
        }
      }
    }
  }
}
```

### Filter Operators

The module supports all standard Search API operators:

- `EQUAL`, `NOT_EQUAL`
- `GREATER_THAN`, `GREATER_THAN_OR_EQUAL`
- `SMALLER_THAN`, `SMALLER_THAN_OR_EQUAL`
- `BETWEEN`, `NOT_BETWEEN`
- `IN`, `NOT_IN`
- `CONTAINS`, `STARTS_WITH`, `ENDS_WITH`
- `LIKE`, `NOT_LIKE`, `REGEXP`
- `IS_NULL`, `IS_NOT_NULL`

### Complex Filtering

You can create complex filter conditions using nested groups:

```graphql
filter: {
  conjunction: AND
  conditions: [
    {
      field: "status"
      operator: EQUAL
      value: ["1"]
    }
  ]
  groups: [
    {
      conjunction: OR
      conditions: [
        {
          field: "type"
          operator: EQUAL
          value: ["article"]
        },
        {
          field: "type"
          operator: EQUAL
          value: ["page"]
        }
      ]
    }
  ]
}
```

### Faceted Search

The module provides built-in faceting support. Specify field names in the `facets` array to get facet results:

```graphql
facets: ["type", "field_tags"]
```

Each facet in the response includes:
- `field`: The field machine name
- `label`: The field label
- `filter`: Object containing `id` (the raw value) and `label` (human-readable label)
- `count`: Number of results for this facet value
- `results`: Array of entities matching this facet

The `filter.label` is automatically resolved based on the field type:
- **Taxonomy reference fields**: Shows the taxonomy term name
- **Entity bundle fields** (`type`): Shows the content type label
- **Entity reference fields**: Shows the referenced entity's label/title/name
- **Other fields**: Falls back to the raw ID value

## Language Support

The module automatically handles language filtering:
- If no language is specified, it uses the current language
- It looks for language fields in this order: `langcode`, `language`, `lang`, `language_code`
- Results are automatically translated to the requested language when available

## Integration with GraphQL Core Schema

This module is designed to work alongside the `graphql_core_schema` module and provides the same filter and sort syntax for consistency. The main difference is that queries run against Search API indexes instead of direct entity queries, providing:

- Better performance for complex searches
- Full-text search capabilities
- Faceting and aggregation features
- Custom relevance scoring

## Troubleshooting

### No Results Returned

1. Verify the Search API index exists and contains data
2. Check that the index is up to date (reindex if necessary)
3. Ensure proper field configuration in the Search API index
4. Verify language settings match your content

### Facets Not Working

1. Check that faceted fields are properly indexed in Search API
2. Verify field names match those in your Search API index configuration

## Development

### Extending the Module

The module follows Drupal plugin architecture:

- `SearchApiQuery` DataProducer: Handles the main search logic
- `SearchApiQueryExtensions` SchemaExtension: Registers GraphQL resolvers

## License

This module is licensed under the same terms as Drupal core.
