# GraphQL Shield - Quick Start Guide

## 🚀 Quick Installation

### Step 1: Install via Composer
```bash
composer require drupal/graphql
```

Then copy the graphql_shield module to your modules directory (already done).

### Step 2: Enable the Module
```bash
drush en graphql_shield -y
```

### Step 3: Configure Basic Security

**Via Admin UI:**
- Navigate to: **Configuration > Web Services > GraphQL Shield**
- Direct URL: `/admin/config/graphql/shield`

**Recommended starter configuration:**
- ✅ Enable Query Complexity (max depth: 10, max complexity: 1000)
- ✅ Enable Rate Limiting (100/user, 60/IP per minute)
- ✅ Enable Logging
- ⚠️ Disable Introspection (in production)
- ℹ️ Persisted Queries: Development mode ON (for development)
- 🔐 **Authentication**: Require API Key for all requests (production only)

### Step 4: Test Your Setup
```bash
# Test a simple query
curl -X POST http://localhost/graphql \
  -H "Content-Type: application/json" \
  -d '{"query": "{ __typename }"}'
```

## 📊 View Security Dashboard

**Via Admin UI:**
- Navigate to: **Reports > GraphQL Shield**
- Direct URL: `/admin/reports/graphql-shield`

**Dashboard Features:**
- Total requests and success rates (24h)
- Blocked requests statistics
- Top users and IPs by request count
- Recent security events
- Slow query detection
- Error distribution charts

## 🔐 Configure API Key Requirements (Optional)

**Force all GraphQL requests to require API keys for maximum security!**

### What is API Key Requirement?

This feature allows you to **enforce mandatory API key authentication** for all GraphQL requests. When enabled:
- ✅ All requests **must** include a valid `X-API-Key` header
- ❌ Requests without API keys are rejected with 401 error
- ⚠️ Optional: Allow authenticated Drupal users to bypass requirement

### Configuration Methods

#### Via Admin UI (Recommended)

1. Navigate to: **Configuration > Web Services > GraphQL Shield**
   - Direct URL: `/admin/config/graphql/shield`
2. Click on the **"Authentication & API Keys"** tab
3. Enable **"Require API key for all requests"**
4. Optionally enable **"Allow authenticated users without API key"** if you want logged-in users to access without keys
5. Click **"Save configuration"**

#### Via Drush (CLI)

```bash
# Enable API key requirement (strict mode)
ddev drush config:set graphql_shield.settings auth.require_api_key true -y

# Optionally allow authenticated users without keys
ddev drush config:set graphql_shield.settings auth.allow_authenticated_without_key true -y

# Clear cache
ddev drush cr
```

### Behavior Matrix

| Configuration | Anonymous Users | Logged-in Users | With Valid API Key |
|---------------|----------------|-----------------|-------------------|
| Requirement OFF | ✅ Allowed | ✅ Allowed | ✅ Allowed |
| Requirement ON, Exception OFF | ❌ Blocked | ❌ Blocked | ✅ Allowed |
| Requirement ON, Exception ON | ❌ Blocked | ✅ Allowed | ✅ Allowed |

### Testing API Key Requirement

**Without API Key (will be blocked):**
```bash
curl -X POST https://your-site.com/graphql \
  -H "Content-Type: application/json" \
  -d '{"query": "{ __typename }"}'

# Response:
{
  "errors": [{
    "message": "API key required. Include X-API-Key header in your request.",
    "extensions": {"code": "AUTHENTICATION_FAILED"}
  }]
}
```

**With Valid API Key (will succeed):**
```bash
curl -X POST https://your-site.com/graphql \
  -H "Content-Type: application/json" \
  -H "X-API-Key: your-api-key-here" \
  -d '{"query": "{ __typename }"}'

# Response: Your data
```

### Use Cases

**1. Public API (Default)**
- Requirement: OFF
- Perfect for public GraphQL APIs
- API keys optional but validated if provided

**2. Strict API (Production)**
- Requirement: ON, Exception: OFF
- Maximum security - everyone needs API keys
- Recommended for external APIs

**3. Hybrid Access (Mixed)**
- Requirement: ON, Exception: ON
- External apps need keys
- Logged-in users don't need keys
- Good for admin interfaces + external integrations

### Important Notes

- ⚠️ **Generate API keys BEFORE enabling requirement**
- 💡 **Test in development first**
- 🔒 **All authentication failures are logged**
- 📊 **Monitor dashboard for blocked attempts**

## 🔑 Generate Your First API Key

API keys in GraphQL Shield are **user-associated**, meaning each API key authenticates as a specific Drupal user and uses that user's roles and permissions.

1. Navigate to: **Configuration > Web Services > GraphQL Shield > API Keys**
   - Or direct URL: `/admin/config/graphql/shield/api-keys`
2. Click **"Add New API Key"** or visit: `/admin/config/graphql/shield/api-keys/add`
3. Fill in the form:
   - **Key Name**: e.g., "Mobile App Production"
   - **Associated User** (required): Select the user this API key will authenticate as
   - **Custom Rate Limit** (optional): e.g., 200 requests/minute
   - **Expiration Date** (optional): When the key should expire
   - **IP Whitelist** (optional): Restrict to specific IPs (one per line)
4. Click **"Generate API Key"**
5. **⚠️ IMPORTANT**: Copy the API key immediately - you won't see it again!

**Important Notes:**
- The API key will authenticate as the selected user
- All permissions are inherited from the associated user's roles
- Content created via the API key will be authored by the associated user

**Using Your API Key:**
```bash
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -H "X-API-Key: your-generated-api-key-here" \
  -d '{"query": "{ articles { title } }"}'
```

**Testing Without API Key Requirement:**
If you haven't enabled "Require API key", the X-API-Key header is optional but will be validated if provided.

**Managing API Keys:**
- **View Keys**: Navigate to `/admin/config/graphql/shield/api-keys` to see all API keys with their expiration status
- **Edit Keys**: Click "Edit" on any key to modify settings (name, user, rate limit, expiration, IP whitelist)
- **Revoke Keys**: Temporarily disable a key without deleting it
- **Delete Keys**: Permanently remove a key

## 📝 Add Your First Persisted Query

> **⚠️ Important Note**: GraphQL Shield's persisted queries are a **security feature** for query whitelisting, NOT the same as GraphQL module's "Persisted Query Plugins" (which are for performance). See README.md for detailed comparison.

**What are GraphQL Shield Persisted Queries?**
- **Purpose**: Security through query whitelisting
- **Benefit**: Only pre-approved queries can execute (blocks unauthorized queries)
- **Use Case**: Production security, preventing malicious queries
- **Development Mode**: Can be toggled to allow all queries during development

**Adding a Persisted Query:**

1. Navigate to: **Configuration > Web Services > GraphQL Shield > Persisted Queries**
   - Or direct URL: `/admin/config/graphql/shield/persisted-queries`
2. Click **"Add New Persisted Query"** or visit: `/admin/config/graphql/shield/persisted-queries/add`
3. Fill in the form:
   - **Query ID**: e.g., "GetArticles" (alphanumeric and underscores only)
   - **GraphQL Query**: Your complete GraphQL query
   - **Description** (optional): What this query does
4. Click **"Add Persisted Query"**

**Using Persisted Queries:**
```bash
# Option 1: By Query ID
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -d '{"queryId": "GetArticles"}'

# Option 2: The full query will be recognized by its hash
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -d '{"query": "query GetArticles { articles { id title } }"}'
```

**When to Enable Persisted Queries:**
- ✅ **Production**: Enable with Development Mode OFF (strict security)
- ✅ **Staging**: Enable with Development Mode ON (testing allowed queries)
- ⚠️ **Development**: Enable with Development Mode ON (allows all queries)

**Quick Comparison:**
| GraphQL Module APQ | GraphQL Shield Persisted Queries |
|-------------------|----------------------------------|
| Performance optimization | Security whitelisting |
| Automatic, client-driven | Manual, admin-managed |
| Caches query strings | Blocks unauthorized queries |
| Optional enforcement | Strict enforcement (production) |

## 🌐 Manage IP Restrictions

1. Navigate to: **Configuration > Web Services > GraphQL Shield > IP Management**
   - Or direct URL: `/admin/config/graphql/shield/ip-management`
2. Add IP rules:
   - **IP Address**: e.g., 192.168.1.100
   - **Rule Type**: Allow (whitelist) or Block (blacklist)
   - **Reason**: Why this rule was created
3. View existing blocked and allowed IPs


## 🔒 Security Features Matrix

| Feature | Status | Production Ready |
|---------|--------|------------------|
| Query Complexity Analysis | ✅ Implemented | Yes |
| Query Depth Limiting | ✅ Implemented | Yes |
| Rate Limiting | ✅ Implemented | Yes |
| Persisted Queries | ✅ Implemented | Yes |
| API Key Management | ✅ Implemented | Yes |
| JWT Authentication | ✅ Implemented | Yes |
| Introspection Control | ✅ Implemented | Yes |
| Request Size Limits | ✅ Implemented | Yes |
| Input Sanitization | ✅ Implemented | Yes |
| Security Logging | ✅ Implemented | Yes |
| IP Restrictions | ✅ Implemented | Yes |
| DoS Protection | ✅ Implemented | Yes |
| CORS Management | ✅ Implemented | Yes |
| Query Pattern Analysis | ✅ Implemented | Yes |
| Error Sanitization | ✅ Implemented | Yes |
| Security Dashboard | ✅ Implemented | Yes |

## 📋 Admin Navigation

**Configuration Pages:**
- Main Settings: Configuration > Web Services > GraphQL Shield
  - Shield Security tab: `/admin/config/graphql/shield`
  - Persisted Queries tab: `/admin/config/graphql/shield/persisted-queries`
  - API Keys tab: `/admin/config/graphql/shield/api-keys`
  - IP Management tab: `/admin/config/graphql/shield/ip-management`

**Reports & Monitoring:**
- Dashboard: Reports > GraphQL Shield
  - Overview tab: `/admin/reports/graphql-shield`
  - Security Logs tab: `/admin/reports/graphql-shield/logs`

## 📚 Documentation

Full documentation available in `README.md`

## 🧪 Testing Checklist

Before deploying to production:

- [ ] Test rate limiting with multiple requests
- [ ] Verify introspection is disabled
- [ ] Test persisted queries
- [ ] Generate and test API keys
- [ ] Review security dashboard
- [ ] Check logs are being recorded
- [ ] Test IP blocking functionality
- [ ] Verify CORS configuration
- [ ] Test error handling
- [ ] Review all configuration settings

## 🎓 Next Steps

1. **Review Configuration**: Adjust settings for your use case
2. **Set Up Monitoring**: Check dashboard regularly
3. **Add Persisted Queries**: Create your query whitelist
4. **Generate API Keys**: For external applications
5. **Configure IP Rules**: Set up allowlist/blocklist
6. **Test Security**: Verify all protections work
7. **Monitor Logs**: Watch for suspicious activity

## 💡 Pro Tips

1. **Start with Development Mode**: Test thoroughly before strict production mode
2. **Monitor Dashboard Daily**: Catch issues early
3. **Use Persisted Queries**: Best practice for production
4. **Set Appropriate Limits**: Balance security and usability
5. **Enable All Logging Initially**: Understand your traffic patterns
6. **Review Blocked Requests**: Ensure no false positives
7. **Keep API Keys Secure**: Treat like passwords
8. **Regular Security Audits**: Review settings monthly

## 🆘 Support

- Check `README.md` for detailed documentation
- Review logs at `/admin/reports/graphql-shield/logs`
- View dashboard at `/admin/reports/graphql-shield`
- Adjust settings at `/admin/config/graphql/shield`

---

**Your GraphQL API is now protected! 🛡️**
