# GraphQL Shield 🛡️

**Comprehensive security module for GraphQL APIs in Drupal**

GraphQL Shield provides enterprise-grade security features for your Drupal GraphQL APIs, protecting against attacks, abuse, and unauthorized access.

## Features

### 🔒 Query Security
- **Query Complexity Analysis** - Prevents resource-intensive queries
- **Query Depth Limiting** - Limits nesting depth to prevent DoS
- **Query Size Validation** - Restricts query size in bytes
- **Query Pattern Analysis** - Detects and blocks suspicious patterns
- **Alias Flooding Protection** - Prevents alias-based attacks
- **Fragment Depth Control** - Limits fragment nesting

### 🚦 Rate Limiting & Throttling
- **Per-User Rate Limiting** - Limit requests per authenticated user
- **Per-IP Rate Limiting** - Limit requests per IP address
- **Flexible Time Windows** - Configurable time periods
- **Token Bucket Algorithm** - Allow burst requests with sustained limits
- **Custom Rate Limits per API Key** - Different limits for different clients

### 📝 Query Whitelisting
- **Persisted Queries** - Only allow pre-approved queries
- **Query Registry** - Manage allowed queries via admin UI
- **Development Mode** - Allow all queries in development
- **Query Hashing** - Automatic query identification
- **Usage Tracking** - Monitor query usage statistics

### 🔐 Authentication & Authorization
- **API Key Management** - Generate and manage API keys
- **User-Associated API Keys** - Each key authenticates as a specific Drupal user
- **Require API Keys** - Force all requests to include valid API keys
- **JWT Token Validation** - Support for JWT authentication
- **Permission Inheritance** - API keys use the associated user's roles and permissions
- **Field-Level Permissions** - Control access to specific fields
- **API Key Features**:
  - User association (required) - keys authenticate as specific users
  - Expiration dates with status display
  - Usage tracking
  - IP whitelisting per key
  - Custom rate limits per key
  - Edit functionality for all key settings
  - Optional requirement (enforce for all requests)
  - Authenticated user exception (allow logged-in users without keys)

### 👁️ Introspection Control
- **Disable in Production** - Hide schema from unauthorized users
- **Role-Based Access** - Allow introspection only for specific roles
- **Authenticated-Only Mode** - Require authentication for introspection

### 🌐 IP Restrictions
- **IP Blocklist** - Block malicious IP addresses
- **IP Allowlist** - Restrict access to specific IPs
- **Auto-Blocking** - Automatically block IPs after violations
- **Temporary Blocks** - Time-based automatic unblocking
- **Manual Management** - Admin UI for IP management

### 🛡️ DoS/DDoS Protection
- **Query Deduplication** - Detect repeated identical queries
- **Connection Limiting** - Limit concurrent connections per IP
- **Circuit Breaker Pattern** - Temporarily disable API under attack
- **Alias Flooding Detection** - Prevent alias-based DoS
- **Slow Query Detection** - Identify resource-intensive queries

### 📊 Logging & Monitoring
- **Comprehensive Security Logs** - All security events logged
- **Query Logging** - Optional logging of all queries
- **Performance Monitoring** - Track execution times
- **Error Logging** - Detailed error tracking
- **Audit Trail** - Full history of who accessed what
- **Security Dashboard** - Real-time statistics and insights

### 🔄 CORS Management
- **Origin Whitelisting** - Control allowed origins
- **Method Control** - Specify allowed HTTP methods
- **Header Management** - Configure allowed/exposed headers
- **Credentials Support** - Control credential sharing
- **Pattern Matching** - Support for wildcard origins

### 🔒 Security Headers
- **Content Security Policy** - Configurable CSP headers
- **X-Frame-Options** - Clickjacking protection
- **HSTS Support** - HTTP Strict Transport Security
- **X-Content-Type-Options** - MIME type sniffing protection
- **Referrer Policy** - Control referrer information

### 🧹 Input Validation & Sanitization
- **Variable Sanitization** - Clean all input variables
- **XSS Protection** - Filter malicious scripts
- **SQL Injection Prevention** - Detect SQL patterns
- **Type Validation** - Ensure correct data types
- **Null Byte Removal** - Prevent null byte attacks

### ⚠️ Error Handling
- **Error Sanitization** - Remove sensitive information
- **Generic Error Messages** - Don't expose internals
- **Stack Trace Hiding** - Remove stack traces in production
- **Custom Error Codes** - Standardized security error codes
- **Detailed Logging** - Full errors logged for admins

## Installation

### Requirements
- Drupal 9.5+ or Drupal 10+
- GraphQL module (drupal/graphql)
- PHP 8.0+

### Via Composer (Recommended)

```bash
composer require drupal/graphql_shield
```

### Enable the Module

```bash
drush en graphql_shield -y
```

Or via the admin UI:
1. Go to **Extend** (`/admin/modules`)
2. Find **GraphQL Shield**
3. Check the box and click **Install**

## Configuration

### Initial Setup

1. Navigate to **Configuration > GraphQL > GraphQL Shield** (`/admin/config/graphql/shield`)

2. Configure your desired security features:

#### Query Complexity Settings
```yaml
Enable: ✓ Enabled
Maximum Query Depth: 10
Maximum Complexity Score: 1000
```

#### Rate Limiting
```yaml
Enable: ✓ Enabled
Requests per User: 100
Requests per IP: 60
Time Window: 60 seconds
```

#### Persisted Queries
```yaml
Enable: ✓ Enabled (for production)
Development Mode: ✓ Enabled (for development only)
```

#### Introspection
```yaml
Enable: ✗ Disabled (recommended for production)
Allow Authenticated: ✓ Enabled
```

### Require API Keys 

**Enforce mandatory API key authentication for all GraphQL requests - a powerful security feature for production APIs.**

#### Overview

This feature transforms your GraphQL API from optional API key validation to **mandatory enforcement**, ensuring that every request is authenticated and authorized. Perfect for:
- 🔒 Production APIs requiring strict access control
- 📊 Tracking and monitoring all API usage
- 🚫 Preventing unauthorized access
- 🎯 Implementing zero-trust security models

#### Configuration

**Via Admin UI:**
1. Go to **Configuration > Web Services > GraphQL Shield** (`/admin/config/graphql/shield`)
2. Click the **"Authentication & API Keys"** tab
3. Enable **"Require API key for all requests"**
4. Optionally enable **"Allow authenticated users without API key"** to exempt logged-in Drupal users
5. Click **"Save configuration"**

**Via Drush:**
```bash
# Enable strict API key requirement
drush config:set graphql_shield.settings auth.require_api_key true -y

# Optionally allow authenticated users
drush config:set graphql_shield.settings auth.allow_authenticated_without_key true -y

# Clear cache
drush cr
```

#### Behavior Modes

| Mode | Config | Anonymous Users | Authenticated Users | With API Key |
|------|--------|----------------|---------------------|--------------|
| **Open** | requirement: OFF | ✅ Allowed | ✅ Allowed | ✅ Allowed |
| **Strict** | requirement: ON<br>exception: OFF | ❌ Blocked | ❌ Blocked | ✅ Allowed |
| **Hybrid** | requirement: ON<br>exception: ON | ❌ Blocked | ✅ Allowed | ✅ Allowed |

#### Error Responses

**When API Key is Missing:**
```json
{
  "errors": [{
    "message": "API key required. Include X-API-Key header in your request.",
    "extensions": {
      "code": "AUTHENTICATION_FAILED",
      "category": "authentication"
    }
  }]
}
```
**HTTP Status**: 401 Unauthorized

**When API Key is Invalid:**
```json
{
  "errors": [{
    "message": "Invalid API key",
    "extensions": {
      "code": "AUTHENTICATION_FAILED"
    }
  }]
}
```
**HTTP Status**: 401 Unauthorized

#### Use Cases

**1. Public GraphQL API (Default Mode)**
```yaml
auth:
  require_api_key: false
```
- Perfect for open APIs
- API keys optional but validated if provided
- Good for development and testing

**2. Enterprise API (Strict Mode)**
```yaml
auth:
  require_api_key: true
  allow_authenticated_without_key: false
```
- Maximum security
- Every request needs an API key
- Best for external partner integrations
- Recommended for production

**3. Mixed Access API (Hybrid Mode)**
```yaml
auth:
  require_api_key: true
  allow_authenticated_without_key: true
```
- External applications need API keys
- Internal logged-in users don't need keys
- Good for admin interfaces + external APIs
- Flexible access control

#### Security Benefits

1. **Authentication Tracking**: Every request is tied to a specific API key
2. **Granular Control**: Revoke access per application instantly
3. **Usage Monitoring**: Track which applications use your API
4. **Rate Limiting**: Apply custom limits per API key
5. **IP Restrictions**: Combine with IP whitelisting per key
6. **Audit Trail**: All authentication failures logged
7. **Compliance**: Meet security audit requirements

#### Production Deployment

**Migration Checklist:**

1. ✅ **Generate API Keys**: Create keys for all applications
2. ✅ **Distribute Keys**: Securely share with application owners
3. ✅ **Update Clients**: Add `X-API-Key` header to all requests
4. ✅ **Test in Staging**: Enable requirement in staging environment first
5. ✅ **Monitor Logs**: Watch for authentication failures
6. ✅ **Enable in Production**: Turn on requirement when ready
7. ✅ **Monitor Dashboard**: Check for any blocked legitimate requests

**Rollback Plan:**
```bash
# If issues arise, quickly disable requirement
drush config:set graphql_shield.settings auth.require_api_key false -y
drush cr
```

### API Key Management

#### User-Based API Key Model

GraphQL Shield uses a **user-based API key model** where each API key is associated with a specific Drupal user account. This architecture provides:

**Benefits:**
- ✅ **Permission Inheritance**: API keys automatically inherit all permissions from the associated user's roles
- ✅ **Content Authorship**: All content created via API keys is properly attributed to the associated user
- ✅ **Audit Trail**: Every API action is tied to a real user account for compliance and tracking
- ✅ **Granular Control**: Simply adjust the user's roles to change API key permissions
- ✅ **User Context**: API keys operate with full user context (same as if the user was logged in)

**How It Works:**
1. When creating an API key, you select a Drupal user to associate it with
2. When the API key is used, it authenticates the request as that user
3. All permission checks use the user's actual roles and permissions
4. Any content created is authored by the associated user
5. All actions appear in logs as performed by that user

**Example Use Cases:**
- **Mobile App**: Create a user "mobile_app_user" with specific permissions, generate an API key for it
- **Partner Integration**: Create a user "partner_api" with limited access, use for partner's API key
- **Admin Dashboard**: Create an API key associated with an admin user for full access
- **Testing**: Create a test user with test permissions for development API keys

**Generate API keys for external applications:**

1. Go to **Configuration > Web Services > GraphQL Shield > API Keys**
   - Direct URL: `/admin/config/graphql/shield/api-keys`
2. Click **"Add New API Key"** or visit `/admin/config/graphql/shield/api-keys/add`
3. Fill in:
   - **Name**: `My Mobile App`
   - **Associated User** (required): Select the user this key will authenticate as
   - **Custom Rate Limit** (optional): 1000 requests/hour
   - **Expiration** (optional): 1 year from now
   - **IP Whitelist** (optional): `203.0.113.0` (one IP per line)
4. Click **"Generate API Key"**
5. **Important**: Copy the API key immediately - it won't be shown again!

**Managing API Keys:**

The API Keys list (`/admin/config/graphql/shield/api-keys`) displays:
- Key name and prefix (first 8 characters)
- Associated user (username and UID)
- Status (Active, Disabled, or Expired)
- Expiration date (or "Never" if no expiration)
- Action buttons (Edit, Revoke, Delete)

**Available Operations:**
- **View**: See all keys with their associated user and expiration status
- **Edit**: Modify key settings (name, associated user, rate limit, expiration, IP whitelist, enabled status)
- **Revoke**: Temporarily disable a key (can be re-enabled via Edit)
- **Delete**: Permanently remove a key (cannot be undone)

**Best Practice:** Create dedicated Drupal user accounts for each API integration with only the necessary permissions, then associate API keys with these accounts.

### Using API Keys

Include the API key in your GraphQL requests:

```bash
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -H "X-API-Key: your-api-key-here" \
  -d '{"query": "{ articles { title } }"}'
```

#### Testing API Key Requirement

**Test 1: Without API Key (Requirement Disabled)**
```bash
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -d '{"query": "{ __typename }"}'
```
✅ **Result**: Request succeeds

**Test 2: Without API Key (Requirement Enabled)**
```bash
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -d '{"query": "{ __typename }"}'
```
❌ **Result**: 401 Unauthorized
```json
{
  "errors": [{
    "message": "API key required. Include X-API-Key header in your request.",
    "extensions": {"code": "AUTHENTICATION_FAILED"}
  }]
}
```

**Test 3: With Valid API Key**
```bash
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -H "X-API-Key: your-valid-api-key" \
  -d '{"query": "{ __typename }"}'
```
✅ **Result**: Request succeeds

**Test 4: With Invalid API Key**
```bash
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -H "X-API-Key: invalid-key-xyz" \
  -d '{"query": "{ __typename }"}'
```
❌ **Result**: 401 Unauthorized
```json
{
  "errors": [{
    "message": "Invalid API key",
    "extensions": {"code": "AUTHENTICATION_FAILED"}
  }]
}
```

### Persisted Queries

#### Adding a Persisted Query

1. Go to **Configuration > Web Services > GraphQL Shield > Persisted Queries**
   - Direct URL: `/admin/config/graphql/shield/persisted-queries`
2. Click **"Add New Persisted Query"** or visit `/admin/config/graphql/shield/persisted-queries/add`
3. Fill in:
   - **Query ID**: `GetArticles` (alphanumeric and underscores only)
   - **Query**:
     ```graphql
     query GetArticles {
       articles {
         id
         title
         body
       }
     }
     ```
   - **Description**: `Fetch all articles for homepage`
4. Click **"Add Persisted Query"**

#### Using Persisted Queries

**Option 1: By Query ID**
```bash
curl -X POST https://example.com/graphql \
  -H "Content-Type: application/json" \
  -d '{"queryId": "GetArticles"}'
```

**Option 2: By Query Hash**
The module will automatically recognize the query hash if you send the full query.

#### Key Differences: GraphQL Shield vs. GraphQL Module Persisted Queries

**Important**: GraphQL Shield's persisted queries are **NOT** the same as the Drupal GraphQL module's "Persisted Query Plugins". They serve different purposes and can work together.

**GraphQL Shield - Persisted Queries (Security Feature)**

GraphQL Shield's persisted queries are a **security and access control mechanism**:

- **Primary Purpose**: Query whitelisting and access control
- **How it works**: Only pre-approved queries stored in the database are allowed to execute
- **Enforcement**: Strict blocking of non-whitelisted queries in production mode
- **Management**: Manual via admin UI (`/admin/config/graphql/shield/persisted-queries`)
- **Use Case**: Prevent malicious or unwanted queries from running
- **Development Mode**: Can be enabled to allow all queries during development
- **Security Benefits**:
  - Prevents arbitrary query execution
  - Reduces attack surface
  - Protects against query-based attacks
  - Provides audit trail of allowed operations
  - Usage tracking and monitoring

**GraphQL Module - Persisted Query Plugins (Performance Feature)**

The base Drupal GraphQL module's persisted query plugins are a **performance optimization** based on Automatic Persisted Queries (APQ):

- **Primary Purpose**: Reduce network overhead and improve performance
- **How it works**: Clients send query hashes instead of full query strings; server caches queries
- **Enforcement**: Optional - clients can still send full queries if hash not found
- **Management**: Automatic, client-driven (queries are cached on first use)
- **Use Case**: Speed up query execution and reduce bandwidth
- **Performance Benefits**:
  - Smaller request payloads
  - Reduced bandwidth usage
  - Faster query transmission
  - Server-side query caching

**Comparison Table**

| Feature | GraphQL Module (APQ) | GraphQL Shield |
|---------|---------------------|----------------|
| **Primary Purpose** | Performance optimization | Security / Query whitelisting |
| **Enforcement** | Optional (client can send full query) | Strict (blocks non-whitelisted queries) |
| **Management** | Automatic, client-driven | Manual via admin UI |
| **Query Rejection** | No (accepts all valid queries) | Yes (rejects non-whitelisted queries) |
| **Development Mode** | N/A | Yes (allows all queries for testing) |
| **Use Case** | Reduce bandwidth and latency | Prevent malicious queries |
| **Query Storage** | Cache (temporary) | Database (permanent) |
| **Security Benefit** | Minimal | High (attack prevention) |
| **Performance Benefit** | High (reduced payload) | Minimal (faster validation) |

**When to Use Which?**

- **Use GraphQL Shield Persisted Queries** when:
  - You want to restrict which queries can run on your API
  - You need to prevent unauthorized or malicious queries
  - You want to whitelist only approved operations
  - Security is a top priority for production

- **Use GraphQL Module Persisted Query Plugins** when:
  - You want to optimize network performance
  - You have bandwidth constraints
  - You want to reduce request/response sizes
  - Performance is a top priority

- **Use Both Together** for:
  - Maximum security AND performance
  - Production APIs with high security requirements
  - Mobile applications with bandwidth constraints
  - Public APIs that need both protection and speed

**Example: Using Both Together**

1. Add queries to GraphQL Shield's whitelist (security)
2. Enable GraphQL module's APQ plugin (performance)
3. Clients send query hashes (small payload)
4. Server validates against Shield's whitelist (security)
5. Server retrieves cached query (performance)
6. Best of both worlds! 🚀

### IP Management

Block or allow specific IP addresses:

1. Go to **Configuration > Web Services > GraphQL Shield > IP Management**
   - Direct URL: `/admin/config/graphql/shield/ip-management`
2. Add a rule:
   - **IP Address**: `192.168.1.100`
   - **Rule Type**: Block or Allow
   - **Reason**: `Malicious activity detected`
3. Click **"Add Rule"**

Auto-blocking will automatically add IPs that violate security policies.

## Security Dashboard

View real-time security statistics:

Navigate to **Reports > GraphQL Shield** or visit `/admin/reports/graphql-shield`

The dashboard shows:
- Total requests (24h) with success/failure rates
- Blocked requests with reasons
- Error rates and distribution
- Top users and IPs by request count
- Recent security events and blocks
- Slow queries (>5 seconds)
- System health indicators
- Hourly statistics charts

## Permissions

Grant permissions at **People > Permissions** (`/admin/people/permissions`):

- **Administer GraphQL Shield** - Full access to configuration
- **View GraphQL Shield Dashboard** - View security reports
- **Manage Persisted Queries** - Add/edit persisted queries
- **Manage API Keys** - Create/revoke API keys
- **Bypass GraphQL Shield** - Skip all security checks (admin only!)

## Best Practices

### Production Configuration

For production environments:

```yaml
Query Complexity: Enabled
Rate Limiting: Enabled (strict limits)
Persisted Queries: Enabled, Development Mode: Disabled
API Key Requirement: Enabled (strict mode recommended)
Introspection: Disabled
Logging: Enabled
Error Sanitization: Enabled
Auto IP Blocking: Enabled
```

**Production API Key Setup:**
```yaml
auth:
  require_api_key: true
  allow_authenticated_without_key: false
```

### Development Configuration

For development:

```yaml
Query Complexity: Enabled (relaxed limits)
Rate Limiting: Enabled (generous limits)
Persisted Queries: Enabled, Development Mode: Enabled
API Key Requirement: Disabled (for easy testing)
Introspection: Enabled
Logging: Enabled
Error Sanitization: Disabled (for debugging)
Auto IP Blocking: Disabled
```

**Development API Key Setup:**
```yaml
auth:
  require_api_key: false  # Optional keys only
```

### Security Checklist

- [ ] Disable introspection in production
- [ ] Enable rate limiting with appropriate limits
- [ ] Use persisted queries in production
- [ ] **Enable API key requirement for production APIs**
- [ ] **Create Drupal users with appropriate permissions for each API integration**
- [ ] **Generate and distribute API keys associated with proper users**
- [ ] Configure CORS properly
- [ ] Enable security logging
- [ ] Set up API keys with correct user associations for external clients
- [ ] Review security dashboard regularly
- [ ] Keep query complexity limits reasonable
- [ ] Enable auto IP blocking
- [ ] Sanitize error messages in production
- [ ] **Monitor authentication failures in logs**
- [ ] **Test API key requirement in staging first**
- [ ] **Regularly audit API key permissions via their associated users**

## Monitoring & Maintenance

### View Logs

Check security logs regularly:
```bash
drush graphql-shield:logs --severity=error
```

### Clean Up Old Logs

Logs are retained for 30 days by default. Clean up manually:
```bash
drush graphql-shield:cleanup-logs --days=30
```

### Reset Rate Limits

If needed, reset rate limits for a user or IP:
```bash
drush graphql-shield:reset-limit --ip=192.168.1.100
drush graphql-shield:reset-limit --uid=123
```

### Circuit Breaker

If the circuit breaker trips, reset it manually:
```bash
drush graphql-shield:reset-circuit
```

## Troubleshooting

### "Rate limit exceeded" Error

**Cause**: Too many requests in the time window.

**Solution**:
1. Wait for the time window to reset
2. Or adjust rate limits in configuration
3. Or reset the rate limit manually

### "Query not allowed. Only persisted queries are permitted"

**Cause**: Persisted queries are enabled in strict mode.

**Solution**:
1. Add the query to the persisted queries list
2. Or enable development mode (not recommended for production)
3. Or disable persisted queries

### "Introspection queries are not allowed"

**Cause**: Introspection is disabled.

**Solution**:
1. Enable introspection for authenticated users
2. Or grant specific roles introspection access
3. Or temporarily enable introspection

### High Memory Usage

**Cause**: Logging all queries can be memory-intensive.

**Solution**:
1. Disable "Log all queries"
2. Only log errors and slow queries
3. Reduce log retention period
4. Enable query deduplication

## API Reference

### Rate Limit Headers

GraphQL Shield adds rate limit headers to responses:

```
X-RateLimit-Limit: 100
X-RateLimit-Remaining: 95
X-RateLimit-Reset: 1640000000
```

### Error Codes

Standard error codes returned:

- `RATE_LIMIT_EXCEEDED` - Too many requests
- `QUERY_TOO_COMPLEX` - Query exceeds complexity limit
- `QUERY_TOO_DEEP` - Query nesting too deep
- `QUERY_TOO_LARGE` - Query size exceeds limit
- `IP_BLOCKED` - IP address is blocked
- `AUTHENTICATION_FAILED` - Invalid credentials
- `ACCESS_DENIED` - Insufficient permissions
- `INVALID_TOKEN` - JWT token invalid or expired
- `PERSISTED_QUERY_NOT_FOUND` - Query not in whitelist
- `INTROSPECTION_DISABLED` - Introspection not allowed
- `DOS_THREAT_DETECTED` - Suspicious patterns detected
- `QUERY_BLOCKED` - Query matches blocked patterns

## Performance Considerations

GraphQL Shield is designed for minimal performance impact:

- **Query complexity analysis**: ~1-2ms per query
- **Rate limiting**: ~0.5ms per query (uses caching)
- **Persisted queries**: ~0.3ms per query (cached)
- **Logging**: ~1-3ms per query (if enabled)

For high-traffic sites:
- Use Redis/Memcache for caching
- Disable "log all queries"
- Use persisted queries to skip validation
- Enable query deduplication

## Contributing

Contributions are welcome! Please:

1. Fork the repository
2. Create a feature branch
3. Follow Drupal coding standards
4. Submit a pull request

## Support

- **Issue Queue**: https://www.drupal.org/project/issues/graphql_shield
- **Documentation**: https://www.drupal.org/docs/contributed-modules/graphql-shield

## License

This module is licensed under the GNU General Public License v2.0 or later.

## Credits

- **Maintainer**: [Mustapha ghazali](https://www.drupal.org/u/gmustapha) 
- **Site**: [ghazali.cc](https://ghazali.cc/) 
- **Blog**: [devopsguy](https://devopsguy.substack.com/)

## Changelog

### 1.0.0
- Initial release
- Full security feature set
- Admin UI and dashboard
- Comprehensive documentation

---

**Stay secure! 🛡️**
