# GraphQL Shield - Module Structure

## Directory Tree
```
graphql_shield/
│
├── 📄 graphql_shield.info.yml          # Module definition
├── 📄 graphql_shield.module            # Hooks (help, theme)
├── 📄 graphql_shield.install           # Install/uninstall + DB schema
├── 📄 graphql_shield.services.yml      # Service definitions (14 services)
├── 📄 graphql_shield.routing.yml       # 8 admin routes
├── 📄 graphql_shield.permissions.yml   # 5 permissions
├── 📄 composer.json                    # Package metadata
├── 📄 README.md                        # Full documentation (400+ lines)
├── 📄 INSTALL.md                       # Quick start guide
├── 📄 SUMMARY.md                       # Implementation summary
│
├── 📁 src/
│   │
│   ├── 📁 Service/                     # Security Services
│   │   ├── 📄 QueryComplexityAnalyzer.php
│   │   ├── 📄 RateLimiter.php
│   │   ├── 📄 PersistedQueryManager.php
│   │   ├── 📄 AuthenticationManager.php
│   │   ├── 📄 IntrospectionController.php
│   │   ├── 📄 RequestValidator.php
│   │   ├── 📄 InputSanitizer.php
│   │   ├── 📄 SecurityLogger.php
│   │   ├── 📄 IpRestrictor.php
│   │   ├── 📄 DosProtector.php
│   │   ├── 📄 CorsManager.php
│   │   ├── 📄 QueryAnalyzer.php
│   │   ├── 📄 ErrorHandler.php
│   │   └── 📄 SecurityDashboard.php
│   │
│   ├── 📁 Middleware/
│   │   └── 📄 GraphQLSecurityMiddleware.php  # Main security pipeline
│   │
│   ├── 📁 Controller/                  # Admin UI Controllers
│   │   ├── 📄 DashboardController.php
│   │   ├── 📄 LogsController.php
│   │   ├── 📄 PersistedQueriesController.php
│   │   └── 📄 ApiKeysController.php
│   │
│   └── 📁 Form/                        # Configuration Forms
│       ├── 📄 GraphQLShieldSettingsForm.php
│       └── 📄 IpManagementForm.php
│
└── 📁 config/
    └── 📁 schema/
        └── 📄 graphql_shield.schema.yml  # Config schema
```

## Database Schema (6 Tables)

```
┌─────────────────────────────────────┐
│   graphql_shield_logs               │  Security Events
├─────────────────────────────────────┤
│ • id, timestamp, uid, ip_address    │
│ • event_type, severity, query       │
│ • variables, message, execution_time│
│ • complexity_score, data            │
└─────────────────────────────────────┘

┌─────────────────────────────────────┐
│   graphql_shield_rate_limits        │  Rate Tracking
├─────────────────────────────────────┤
│ • identifier, identifier_type       │
│ • request_count, window_start/end   │
│ • last_request                      │
└─────────────────────────────────────┘

┌─────────────────────────────────────┐
│   graphql_shield_persisted_queries  │  Query Whitelist (Security)
├─────────────────────────────────────┤
│ • query_hash, query_id, query       │
│ • description, enabled, created     │
│ • usage_count                       │
│                                     │
│ ⚠️ NOTE: This is for SECURITY       │
│ (query whitelisting), NOT for       │
│ performance optimization like       │
│ GraphQL module's APQ plugin         │
└─────────────────────────────────────┘

┌─────────────────────────────────────┐
│   graphql_shield_api_keys           │  API Keys
├─────────────────────────────────────┤
│ • key_hash, key_prefix, name        │
│ • uid (associated user), rate_limit │
│ • ip_whitelist, enabled, created    │
│ • expires, last_used, usage_count   │
│                                     │
│ ⚠️ NOTE: API keys authenticate as   │
│ the associated user (uid) and use   │
│ that user's roles and permissions.  │
│ The 'roles' column exists in DB but │
│ is deprecated and unused.           │
└─────────────────────────────────────┘

┌─────────────────────────────────────┐
│   graphql_shield_ip_rules           │  IP Rules
├─────────────────────────────────────┤
│ • ip_address, rule_type (allow/block)│
│ • reason, created, expires          │
│ • auto_created                      │
└─────────────────────────────────────┘
```

## Service Dependencies Graph

```
                    ┌─────────────────────────────────┐
                    │  GraphQLSecurityMiddleware      │
                    │  (HTTP Middleware - Priority 250)│
                    └────────────┬────────────────────┘
                                 │
         ┌───────────────────────┼───────────────────────┐
         │                       │                       │
         ▼                       ▼                       ▼
┌─────────────────┐   ┌─────────────────┐   ┌─────────────────┐
│ RateLimiter     │   │ IpRestrictor    │   │ DosProtector    │
│ • checkLimit    │   │ • isAllowed     │   │ • checkThreat   │
└─────────────────┘   └─────────────────┘   └─────────────────┘
         │                       │                       │
         └───────────────────────┼───────────────────────┘
                                 │
         ┌───────────────────────┼───────────────────────┐
         │                       │                       │
         ▼                       ▼                       ▼
┌─────────────────┐   ┌──────────────────┐   ┌─────────────────┐
│ComplexityAnalyze│   │PersistedQuery    │   │ AuthManager     │
│ • analyze       │   │ • validateQuery  │   │ • validateKey   │
└─────────────────┘   └──────────────────┘   └─────────────────┘
         │                       │                       │
         └───────────────────────┼───────────────────────┘
                                 │
         ┌───────────────────────┼───────────────────────┐
         │                       │                       │
         ▼                       ▼                       ▼
┌─────────────────┐   ┌──────────────────┐   ┌─────────────────┐
│RequestValidator │   │ QueryAnalyzer    │   │ ErrorHandler    │
│ • validate      │   │ • analyze        │   │ • sanitize      │
└─────────────────┘   └──────────────────┘   └─────────────────┘
         │                       │                       │
         └───────────────────────┼───────────────────────┘
                                 │
                                 ▼
                    ┌─────────────────────────────────┐
                    │      SecurityLogger             │
                    │  • logEvent, logQuery, logError │
                    └─────────────────────────────────┘
```

## Request Flow Diagram

```
┌─────────────────────────────────────────────────────────────────┐
│  INCOMING HTTP REQUEST → /graphql                               │
└────────────────────┬────────────────────────────────────────────┘
                     │
                     ▼
      ┌──────────────────────────────────────┐
      │  GraphQLSecurityMiddleware::handle()  │
      └──────────────┬───────────────────────┘
                     │
    ┌────────────────┴────────────────┐
    │  1. IP Check                    │ → Blocked? → 403 Response
    └────────────────┬────────────────┘
                     │ ✓ Allowed
    ┌────────────────┴────────────────┐
    │  2. Circuit Breaker Check       │ → Open? → 503 Response
    └────────────────┬────────────────┘
                     │ ✓ Closed
    ┌────────────────┴────────────────┐
    │  3. API Key Validation          │ → Invalid? → 401 Response
    └────────────────┬────────────────┘
                     │ ✓ Valid
    ┌────────────────┴────────────────┐
    │  4. Rate Limit Check            │ → Exceeded? → 429 Response
    └────────────────┬────────────────┘
                     │ ✓ Within Limit
    ┌────────────────┴────────────────┐
    │  5. Persisted Query Validation  │ → Not Allowed? → 400 Response
    └────────────────┬────────────────┘
                     │ ✓ Allowed
    ┌────────────────┴────────────────┐
    │  6. Request Size/Structure      │ → Too Large? → 400 Response
    └────────────────┬────────────────┘
                     │ ✓ Valid
    ┌────────────────┴────────────────┐
    │  7. Input Sanitization          │ → Sanitize Variables
    └────────────────┬────────────────┘
                     │ ✓ Sanitized
    ┌────────────────┴────────────────┐
    │  8. Introspection Check         │ → Not Allowed? → 403 Response
    └────────────────┬────────────────┘
                     │ ✓ Allowed
    ┌────────────────┴────────────────┐
    │  9. Query Complexity Analysis   │ → Too Complex? → 400 Response
    └────────────────┬────────────────┘
                     │ ✓ Within Limits
    ┌────────────────┴────────────────┐
    │  10. DoS Pattern Detection      │ → Threat? → 403 Response
    └────────────────┬────────────────┘
                     │ ✓ Safe
    ┌────────────────┴────────────────┐
    │  11. Advanced Query Analysis    │ → Suspicious? → 400 Response
    └────────────────┬────────────────┘
                     │ ✓ Safe
    ┌────────────────┴────────────────┐
    │  ✅ ALL CHECKS PASSED            │
    └────────────────┬────────────────┘
                     │
    ┌────────────────┴────────────────┐
    │  Process GraphQL Query          │
    └────────────────┬────────────────┘
                     │
    ┌────────────────┴────────────────┐
    │  Log Query + Add Headers        │
    └────────────────┬────────────────┘
                     │
                     ▼
┌─────────────────────────────────────────────────────────────────┐
│  RETURN RESPONSE with Rate Limit Headers                        │
└─────────────────────────────────────────────────────────────────┘
```

## Admin UI Map

```
┌────────────────────────────────────────────────────────────────┐
│  Configuration > GraphQL                                        │
└────────────────────────────────────────────────────────────────┘
                     │
        ┌────────────┴──────────────────────────┐
        │                                       │
        ▼                                       ▼
┌───────────────────────┐         ┌───────────────────────────┐
│ GraphQL Shield        │         │ GraphQL Shield            │
│ /admin/config/graphql/│         │ /admin/reports/graphql-   │
│ shield                │         │ shield                    │
├───────────────────────┤         ├───────────────────────────┤
│ • Query Complexity    │         │ • Dashboard (Overview)    │
│ • Rate Limiting       │         │ • Top Users/IPs           │
│ • Persisted Queries   │         │ • Recent Blocks           │
│ • Introspection       │         │ • Slow Queries            │
│ • Request Limits      │         │ • Error Distribution      │
│ • IP Restrictions     │         │ • Hourly Stats            │
│ • DoS Protection      │         └───────────────────────────┘
│ • Logging             │                     │
│ • CORS                │         ┌───────────┴───────────┐
└───────────────────────┘         │                       │
        │                         ▼                       ▼
        │              ┌──────────────────┐   ┌─────────────────┐
        ├─────────────>│ Security Logs    │   │ More Reports... │
        │              │ /admin/reports/  │   └─────────────────┘
        │              │ graphql-shield/  │
        │              │ logs             │
        │              └──────────────────┘
        │
        ├─────────────>┌──────────────────┐
        │              │ Persisted Queries│
        │              │ /admin/config/   │
        │              │ graphql/shield/  │
        │              │ persisted-queries│
        │              └──────────────────┘
        │
        ├─────────────>┌──────────────────┐
        │              │ API Keys         │
        │              │ /admin/config/   │
        │              │ graphql/shield/  │
        │              │ api-keys         │
        │              └──────────────────┘
        │
        └─────────────>┌──────────────────┐
                       │ IP Management    │
                       │ /admin/config/   │
                       │ graphql/shield/  │
                       │ ip-management    │
                       └──────────────────┘
```

## Feature Coverage Matrix

## Key Service Descriptions

### PersistedQueryManager (Security Whitelisting)

**⚠️ IMPORTANT**: This is NOT the same as GraphQL module's "Persisted Query Plugins" (APQ)

**Purpose**: Security through query whitelisting (NOT performance optimization)

**What it does**:
- Maintains a database of pre-approved GraphQL queries
- Validates incoming queries against the whitelist
- Blocks queries that are not in the whitelist (production mode)
- Provides development mode to allow all queries during development
- Tracks query usage statistics

**Key Methods**:
- `isEnabled()` - Check if persisted queries are enforced
- `isDevelopmentMode()` - Check if development mode is active
- `validateQuery($query, $query_id)` - Validate if a query is allowed
- `getQuery($identifier)` - Retrieve a persisted query by hash or ID
- `addQuery($query_id, $query, $description)` - Add query to whitelist
- `removeQuery($id)` - Remove query from whitelist

**Comparison with GraphQL Module APQ**:

| GraphQL Shield PersistedQueryManager | GraphQL Module APQ Plugin |
|-------------------------------------|---------------------------|
| **Security**: Query whitelisting | **Performance**: Caching |
| Blocks non-whitelisted queries | Accepts all valid queries |
| Database storage (permanent) | Cache storage (temporary) |
| Admin-managed via UI | Client-driven, automatic |
| Strict enforcement | Optional optimization |
| Development mode toggle | No enforcement mode |

**Database Table**: `graphql_shield_persisted_queries`
- Stores query hash (SHA256), query ID, full query text
- Tracks enabled/disabled status, usage count
- Unique constraint on query_hash and query_id

**Usage Example**:
```php
// Check if query is allowed
$result = $persistedQueryManager->validateQuery($query, $queryId);
if (!$result['allowed']) {
  // Block the request - query not in whitelist
  throw new AccessDeniedHttpException($result['error']);
}
// Allow the request - query is whitelisted
$allowedQuery = $result['query'];
```

### Other Key Services

**QueryComplexityAnalyzer**: Analyzes query depth and complexity to prevent resource-intensive queries

**RateLimiter**: Enforces rate limits per user and per IP address using token bucket algorithm

**AuthenticationManager**: Validates API keys and JWT tokens

**IpRestrictor**: Manages IP allowlist and blocklist rules

**DosProtector**: Detects and prevents DoS attacks using circuit breaker pattern

**SecurityLogger**: Logs all security events to database for audit trail

**SecurityDashboard**: Generates statistics and insights from security logs

