<?php

namespace Drupal\graphql_shield\Controller;

use Drupal\user\Entity\User;
use Drupal\Core\Url;
use Drupal\Core\Controller\ControllerBase;
use Drupal\graphql_shield\Service\AuthenticationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for API keys management.
 */
class ApiKeysController extends ControllerBase {

  /**
   * The authentication manager.
   *
   * @var \Drupal\graphql_shield\Service\AuthenticationManager
   */
  protected $authManager;

  /**
   * Constructs an ApiKeysController object.
   */
  public function __construct(AuthenticationManager $auth_manager) {
    $this->authManager = $auth_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.auth_manager')
    );
  }

  /**
   * Lists API keys.
   */
  public function list() {
    $keys = $this->authManager->listApiKeys();

    $build = [];

    $build['intro'] = [
      '#markup' => '<p>' . $this->t('API keys provide secure authentication for GraphQL requests. Keep your API keys secret and never commit them to version control.') . '</p>',
    ];

    $build['add_link'] = [
      '#markup' => '<p><a href="/admin/config/graphql/shield/api-keys/add">' . $this->t('Generate New API Key') . '</a></p>',
    ];

    if (empty($keys)) {
      $build['empty'] = [
        '#markup' => '<p>' . $this->t('No API keys found.') . '</p>',
      ];
    }
    else {
      $rows = [];
      foreach ($keys as $key) {
        $status = $key['enabled'] ? $this->t('Active') : $this->t('Disabled');

        if ($key['expires'] && $key['expires'] < time()) {
          $status = $this->t('Expired');
        }

        // Build operations links.
        $operations = [];
        $operations[] = [
          '#type' => 'link',
          '#title' => $this->t('Edit'),
          '#url' => Url::fromRoute('graphql_shield.api_keys.edit', ['id' => $key['id']]),
          '#attributes' => ['class' => ['button', 'button--small']],
        ];
        if ($key['enabled']) {
          $operations[] = [
            '#type' => 'link',
            '#title' => $this->t('Revoke'),
            '#url' => Url::fromRoute('graphql_shield.api_keys.revoke', ['id' => $key['id']]),
            '#attributes' => ['class' => ['button', 'button--small']],
          ];
        }
        $operations[] = [
          '#type' => 'link',
          '#title' => $this->t('Delete'),
          '#url' => Url::fromRoute('graphql_shield.api_keys.delete', ['id' => $key['id']]),
          '#attributes' => ['class' => ['button', 'button--small', 'button--danger']],
        ];

        // Load associated user.
        $user = User::load($key['uid']);
        $user_display = $user ? $user->getAccountName() . ' (' . $user->id() . ')' : $this->t('Unknown');

        // Format expiration.
        if ($key['expires']) {
          $expires_display = date('Y-m-d H:i:s', $key['expires']);
          // Check if expired.
          if ($key['expires'] < time()) {
            $expires_display .= ' ' . $this->t('(Expired)');
          }
        }
        else {
          $expires_display = $this->t('Never');
        }

        $rows[] = [
          $key['key_prefix'] . '...',
          $key['name'],
          $user_display,
          $status,
          $expires_display,
          number_format($key['usage_count']),
          date('Y-m-d H:i:s', $key['created']),
          $key['last_used'] ? date('Y-m-d H:i:s', $key['last_used']) : $this->t('Never'),
          ['data' => $operations],
        ];
      }

      $build['table'] = [
        '#theme' => 'table',
        '#header' => [
          $this->t('Key'),
          $this->t('Name'),
          $this->t('User'),
          $this->t('Status'),
          $this->t('Expires'),
          $this->t('Usage Count'),
          $this->t('Created'),
          $this->t('Last Used'),
          $this->t('Operations'),
        ],
        '#rows' => $rows,
      ];
    }

    return $build;
  }

}
