<?php

namespace Drupal\graphql_shield\Controller;

use Drupal\Core\Url;
use Drupal\Core\Controller\ControllerBase;
use Drupal\graphql_shield\Service\SecurityDashboard;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the GraphQL Shield dashboard.
 */
class DashboardController extends ControllerBase {

  /**
   * The security dashboard service.
   *
   * @var \Drupal\graphql_shield\Service\SecurityDashboard
   */
  protected $dashboard;

  /**
   * Constructs a DashboardController object.
   */
  public function __construct(SecurityDashboard $dashboard) {
    $this->dashboard = $dashboard;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.dashboard')
    );
  }

  /**
   * Dashboard page.
   */
  public function index() {
    // Get dashboard data for last 24 hours.
    $data = $this->dashboard->getDashboardData(86400);

    $build = [];

    // Overview statistics.
    $build['overview'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Overview (Last 24 Hours)'),
    ];

    $build['overview']['stats'] = [
      '#theme' => 'table',
      '#header' => [$this->t('Metric'), $this->t('Value')],
      '#rows' => [
        [$this->t('Total Requests'), number_format($data['overview']['total_requests'])],
        [$this->t('Successful Queries'), number_format($data['overview']['successful_queries'])],
        [$this->t('Blocked Requests'), number_format($data['overview']['blocked_requests'])],
        [$this->t('Errors'), number_format($data['overview']['errors'])],
        [$this->t('Unique Users'), number_format($data['overview']['unique_users'])],
        [$this->t('Unique IPs'), number_format($data['overview']['unique_ips'])],
        [$this->t('Avg Response Time'), $data['overview']['avg_response_time'] . ' s'],
        [$this->t('Active API Keys'), number_format($data['overview']['api_keys_active'])],
        [$this->t('Persisted Queries'), number_format($data['overview']['persisted_queries'])],
        [$this->t('IP Blocks Active'), number_format($data['overview']['ip_blocks'])],
      ],
    ];

    // Top users.
    if (!empty($data['top_users'])) {
      $build['top_users'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Top Users by Request Count'),
      ];

      $rows = [];
      foreach ($data['top_users'] as $user_data) {
        $rows[] = [
          $user_data['uid'],
          number_format($user_data['request_count']),
        ];
      }

      $build['top_users']['table'] = [
        '#theme' => 'table',
        '#header' => [$this->t('User ID'), $this->t('Requests')],
        '#rows' => $rows,
      ];
    }

    // Top IPs.
    if (!empty($data['top_ips'])) {
      $build['top_ips'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Top IPs by Request Count'),
      ];

      $rows = [];
      foreach ($data['top_ips'] as $ip_data) {
        $rows[] = [
          $ip_data['ip_address'],
          number_format($ip_data['request_count']),
        ];
      }

      $build['top_ips']['table'] = [
        '#theme' => 'table',
        '#header' => [$this->t('IP Address'), $this->t('Requests')],
        '#rows' => $rows,
      ];
    }

    // Recent blocks.
    if (!empty($data['recent_blocks'])) {
      $build['recent_blocks'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Recent Blocked Requests'),
      ];

      $rows = [];
      foreach ($data['recent_blocks'] as $block) {
        $rows[] = [
          date('Y-m-d H:i:s', $block['timestamp']),
          $block['ip_address'],
          $block['message'],
        ];
      }

      $build['recent_blocks']['table'] = [
        '#theme' => 'table',
        '#header' => [
          $this->t('Time'),
          $this->t('IP Address'),
          $this->t('Reason'),
        ],
        '#rows' => $rows,
      ];
    }

    // Slow queries.
    if (!empty($data['slow_queries'])) {
      $build['slow_queries'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Slow Queries'),
      ];

      $rows = [];
      foreach ($data['slow_queries'] as $query) {
        $rows[] = [
          date('Y-m-d H:i:s', $query['timestamp']),
          round($query['execution_time'], 3) . ' s',
          substr($query['query'], 0, 100) . '...',
        ];
      }

      $build['slow_queries']['table'] = [
        '#theme' => 'table',
        '#header' => [
          $this->t('Time'),
          $this->t('Execution Time'),
          $this->t('Query'),
        ],
        '#rows' => $rows,
      ];
    }

    $build['actions'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Quick Actions'),
    ];

    $build['actions']['links'] = [
      '#theme' => 'item_list',
      '#items' => [
        [
          '#markup' => $this->t('<a href="@url">View Full Logs</a>', [
            '@url' => '/admin/reports/graphql-shield/logs',
          ]),
        ],
        [
          '#markup' => $this->t('<a href="@url">Manage IP Restrictions</a>', [
            '@url' => '/admin/config/graphql/shield/ip-management',
          ]),
        ],
        [
          '#markup' => $this->t('<a href="@url">Configure Settings</a>', [
            '@url' => '/admin/config/graphql/shield',
          ]),
        ],

      ],

    ];

    // Add clear logs button if user has permission.
    if ($this->currentUser()->hasPermission('clear graphql shield logs')) {
      $build['actions']['clear_logs'] = [
        '#type' => 'link',
        '#title' => $this->t('Clear All Logs'),
        '#url' => Url::fromRoute('graphql_shield.logs.clear'),
        '#attributes' => [
          'class' => ['button', 'button--danger'],
        ],
        '#prefix' => '<div style="margin-top: 10px;">',
        '#suffix' => '</div>',
      ];
    }

    return $build;
  }

}
