<?php

namespace Drupal\graphql_shield\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\graphql_shield\Service\SecurityLogger;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for security logs.
 */
class LogsController extends ControllerBase {

  /**
   * The security logger service.
   *
   * @var \Drupal\graphql_shield\Service\SecurityLogger
   */
  protected $securityLogger;

  /**
   * Constructs a LogsController object.
   */
  public function __construct(SecurityLogger $security_logger) {
    $this->securityLogger = $security_logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.security_logger')
    );
  }

  /**
   * Logs page.
   */
  public function index(Request $request) {
    $filters = [
      'event_type' => $request->query->get('event_type'),
      'severity' => $request->query->get('severity'),
    ];

    $logs = $this->securityLogger->getLogs(100, array_filter($filters));

    $build = [];

    $build['filters'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Filters'),
    ];

    $build['filters']['event_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Event Type'),
      '#options' => [
        '' => $this->t('- All -'),
        'query' => $this->t('Query'),
        'blocked' => $this->t('Blocked'),
        'error' => $this->t('Error'),
      ],
      '#default_value' => $filters['event_type'] ?? '',
    ];

    $build['filters']['severity'] = [
      '#type' => 'select',
      '#title' => $this->t('Severity'),
      '#options' => [
        '' => $this->t('- All -'),
        'info' => $this->t('Info'),
        'warning' => $this->t('Warning'),
        'error' => $this->t('Error'),
        'critical' => $this->t('Critical'),
      ],
      '#default_value' => $filters['severity'] ?? '',
    ];

    $build['logs'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Security Logs'),
    ];

    if (empty($logs)) {
      $build['logs']['empty'] = [
        '#markup' => $this->t('No logs found.'),
      ];
    }
    else {
      $rows = [];
      foreach ($logs as $log) {
        $rows[] = [
          date('Y-m-d H:i:s', $log['timestamp']),
          $log['event_type'],
          $log['severity'],
          $log['ip_address'],
          $log['uid'],
          substr($log['message'] ?? '', 0, 100),
        ];
      }

      $build['logs']['table'] = [
        '#theme' => 'table',
        '#header' => [
          $this->t('Timestamp'),
          $this->t('Event Type'),
          $this->t('Severity'),
          $this->t('IP Address'),
          $this->t('User ID'),
          $this->t('Message'),
        ],
        '#rows' => $rows,
      ];
    }

    return $build;
  }

}
