<?php

namespace Drupal\graphql_shield\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\graphql_shield\Service\PersistedQueryManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for persisted queries management.
 */
class PersistedQueriesController extends ControllerBase {

  /**
   * The persisted query manager.
   *
   * @var \Drupal\graphql_shield\Service\PersistedQueryManager
   */
  protected $persistedQueryManager;

  /**
   * Constructs a PersistedQueriesController object.
   */
  public function __construct(PersistedQueryManager $persisted_query_manager) {
    $this->persistedQueryManager = $persisted_query_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.persisted_query')
    );
  }

  /**
   * Lists persisted queries.
   */
  public function list() {
    $queries = $this->persistedQueryManager->listQueries();

    $build = [];

    $build['intro'] = [
      '#markup' => '<p>' . $this->t('Persisted queries allow you to whitelist specific GraphQL queries. When enabled in strict mode, only these queries will be allowed.') . '</p>',
    ];

    $build['add_link'] = [
      '#markup' => '<p><a href="/admin/config/graphql/shield/persisted-queries/add">' . $this->t('Add New Persisted Query') . '</a></p>',
    ];

    if (empty($queries)) {
      $build['empty'] = [
        '#markup' => '<p>' . $this->t('No persisted queries found.') . '</p>',
      ];
    }
    else {
      $rows = [];
      foreach ($queries as $query) {
        $rows[] = [
          $query['query_id'],
          substr($query['description'] ?? '', 0, 100),
          $query['enabled'] ? $this->t('Enabled') : $this->t('Disabled'),
          number_format($query['usage_count']),
          date('Y-m-d H:i:s', $query['created']),
          $this->t('Edit | Delete'),
        ];
      }

      $build['table'] = [
        '#theme' => 'table',
        '#header' => [
          $this->t('Query ID'),
          $this->t('Description'),
          $this->t('Status'),
          $this->t('Usage Count'),
          $this->t('Created'),
          $this->t('Operations'),
        ],
        '#rows' => $rows,
      ];
    }

    return $build;
  }

}
