<?php

namespace Drupal\graphql_shield\EventSubscriber;

use Drupal\user\Entity\User;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\graphql_shield\Service\AuthenticationManager;
use Drupal\graphql_shield\Service\ErrorHandler;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber to check API key authentication for GraphQL requests.
 *
 * This runs AFTER Drupal's authentication (priority 300) so we can check
 * if the user is logged in via session cookies.
 */
class ApiKeyAuthenticationSubscriber implements EventSubscriberInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The authentication manager.
   *
   * @var \Drupal\graphql_shield\Service\AuthenticationManager
   */
  protected $authManager;

  /**
   * The error handler.
   *
   * @var \Drupal\graphql_shield\Service\ErrorHandler
   */
  protected $errorHandler;

  /**
   * Cached GraphQL endpoint paths.
   *
   * @var array
   */
  protected $graphqlPaths = NULL;

  /**
   * Constructs a new ApiKeyAuthenticationSubscriber.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountProxyInterface $current_user,
    AuthenticationManager $auth_manager,
    ErrorHandler $error_handler,
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->authManager = $auth_manager;
    $this->errorHandler = $error_handler;
  }

  /**
   * Checks API key authentication for GraphQL requests.
   */
  public function onKernelRequest(RequestEvent $event) {
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();

    // Only check GraphQL requests.
    if (!$this->isGraphqlRequest($request)) {
      return;
    }

    $config = $this->configFactory->get('graphql_shield.settings');
    $require_api_key = $config->get('auth.require_api_key');

    if (!$require_api_key) {
      return;
    }

    $allow_authenticated_without_key = $config->get('auth.allow_authenticated_without_key');
    $api_key = $request->headers->get('X-API-Key');

    // Check if user is authenticated and allowed without key.
    $is_authenticated = $this->currentUser->isAuthenticated();
    $has_exception = $allow_authenticated_without_key && $is_authenticated;

    if (!$has_exception) {
      // API key is required.
      if (!$api_key) {
        $response = new JsonResponse([
          'errors' => [
            [
              'message' => 'API key required. Include X-API-Key header in your request.',
              'extensions' => [
                'code' => $this->errorHandler->getSecurityErrorCode('auth_failed'),
              ],
            ],
          ],
        ], 401);
        $event->setResponse($response);
        return;
      }

      // Validate the provided API key.
      $auth_result = $this->authManager->validateApiKey($api_key);
      if (!$auth_result['valid']) {
        $response = new JsonResponse([
          'errors' => [
            [
              'message' => 'Invalid API key',
              'extensions' => [
                'code' => $this->errorHandler->getSecurityErrorCode('auth_failed'),
              ],
            ],
          ],
        ], 401);
        $event->setResponse($response);
        return;
      }

      // Set the user from the API key.
      $key_data = $auth_result['key_data'];

      // Load the associated user.
      $user = User::load($key_data['uid']);
      if (!$user || $user->isBlocked()) {
        $response = new JsonResponse([
          'errors' => [
            [
              'message' => 'API key associated user not found or blocked',
              'extensions' => [
                'code' => $this->errorHandler->getSecurityErrorCode('auth_failed'),
              ],
            ],
          ],
        ], 401);
        $event->setResponse($response);
        return;
      }

      // Set the user as current user (with their own roles and permissions)
      \Drupal::service('current_user')->setAccount($user);
    }
  }

  /**
   * Checks if the request is a GraphQL request.
   */
  protected function isGraphqlRequest($request) {
    if ($this->graphqlPaths === NULL) {
      $this->graphqlPaths = [];

      try {
        $storage = \Drupal::entityTypeManager()->getStorage('graphql_server');
        $servers = $storage->loadMultiple();

        foreach ($servers as $server) {
          // GraphQL server entities have a public endpoint property.
          $endpoint = $server->endpoint ?? NULL;
          if ($endpoint) {
            $this->graphqlPaths[] = $endpoint;
          }
        }
      }
      catch (\Exception $e) {
        // If we can't load servers, fall back to checking path.
      }
    }

    $path = $request->getPathInfo();

    foreach ($this->graphqlPaths as $graphql_path) {
      if (strpos($path, $graphql_path) === 0) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    // Run AFTER authentication (priority 300) but before access checks
    // (priority 31).
    $events[KernelEvents::REQUEST][] = ['onKernelRequest', 250];
    return $events;
  }

}
