<?php

namespace Drupal\graphql_shield\Form;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\graphql_shield\Service\AuthenticationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for generating a new API key.
 */
class ApiKeyAddForm extends FormBase {

  /**
   * The authentication manager.
   *
   * @var \Drupal\graphql_shield\Service\AuthenticationManager
   */
  protected $authManager;

  /**
   * Constructs an ApiKeyAddForm object.
   */
  public function __construct(AuthenticationManager $auth_manager) {
    $this->authManager = $auth_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.auth_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'graphql_shield_api_key_add';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Key Name'),
      '#description' => $this->t('A descriptive name for this API key (e.g., "Mobile App", "Partner Integration").'),
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['uid'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => 'user',
      '#title' => $this->t('Associated User'),
      '#description' => $this->t('Select the user that will be associated with this API key. This user will be the author of any content created through this API key.'),
      '#required' => TRUE,
      '#default_value' => User::load(\Drupal::currentUser()->id()),
      '#selection_settings' => [
        'include_anonymous' => FALSE,
      ],
    ];

    $form['rate_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Custom Rate Limit'),
      '#description' => $this->t('Optional: Set a custom rate limit for this key (requests per minute). Leave empty to use default.'),
      '#min' => 1,
    ];

    $form['expires'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Expiration Date'),
      '#description' => $this->t('Optional: Set when this key should expire. Leave empty for no expiration.'),
    ];

    $form['ip_whitelist'] = [
      '#type' => 'textarea',
      '#title' => $this->t('IP Whitelist'),
      '#description' => $this->t('Optional: Restrict this key to specific IP addresses (one per line). Leave empty to allow all IPs.'),
      '#rows' => 5,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate API Key'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('graphql_shield.api_keys'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $name = $form_state->getValue('name');

    // Get the selected user.
    $user_entity = $form_state->getValue('uid');
    $uid = $user_entity instanceof User ? $user_entity->id() : $user_entity;

    $options = [];

    // Custom rate limit.
    if ($rate_limit = $form_state->getValue('rate_limit')) {
      $options['rate_limit'] = $rate_limit;
    }

    // Expiration.
    if ($expires = $form_state->getValue('expires')) {
      if ($expires instanceof DrupalDateTime) {
        $options['expires'] = $expires->getTimestamp();
      }
    }

    // IP whitelist.
    if ($ip_whitelist = $form_state->getValue('ip_whitelist')) {
      $ips = array_filter(array_map('trim', explode("\n", $ip_whitelist)));
      if (!empty($ips)) {
        $options['ip_whitelist'] = $ips;
      }
    }

    // Generate the API key.
    $result = $this->authManager->generateApiKey($name, $uid, $options);

    // Show the key to the user (only time they'll see it!).
    $this->messenger()->addStatus($this->t('API Key generated successfully!'));
    $this->messenger()->addWarning($this->t("<strong>Important:</strong> Copy this API key now. You won't be able to see it again!"));
    $this->messenger()->addMessage($this->t('<div style="background: #f0f0f0; padding: 15px; margin: 10px 0; font-family: monospace; font-size: 14px; word-break: break-all;"><strong>API Key:</strong><br/>@key</div>', [
      '@key' => $result['key'],
    ]));

    // Redirect back to the list.
    $form_state->setRedirect('graphql_shield.api_keys');
  }

}
