<?php

namespace Drupal\graphql_shield\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\graphql_shield\Service\AuthenticationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for deleting an API key.
 */
class ApiKeyDeleteForm extends ConfirmFormBase {

  /**
   * The authentication manager.
   *
   * @var \Drupal\graphql_shield\Service\AuthenticationManager
   */
  protected $authManager;

  /**
   * The API key ID to delete.
   *
   * @var int
   */
  protected $keyId;

  /**
   * The API key data.
   *
   * @var array
   */
  protected $keyData;

  /**
   * Constructs an ApiKeyDeleteForm object.
   */
  public function __construct(AuthenticationManager $auth_manager) {
    $this->authManager = $auth_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.auth_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'graphql_shield_api_key_delete';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $this->keyId = $id;

    // Load the API key data.
    $this->keyData = $this->authManager->getApiKey($id);

    if (!$this->keyData) {
      $this->messenger()->addError($this->t('API key not found.'));
      return $this->redirect('graphql_shield.api_keys');
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to permanently delete the API key "@name"?', [
      '@name' => $this->keyData['name'] ?? 'Unknown',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('<strong>Warning:</strong> This action cannot be undone. The API key will be permanently removed from the database. Key prefix: @prefix<br><br>Usage count: @usage<br>Created: @created<br>Last used: @last_used', [
      '@prefix' => $this->keyData['key_prefix'] ?? 'N/A',
      '@usage' => number_format($this->keyData['usage_count'] ?? 0),
      '@created' => date('Y-m-d H:i:s', $this->keyData['created'] ?? 0),
      '@last_used' => $this->keyData['last_used'] ? date('Y-m-d H:i:s', $this->keyData['last_used']) : $this->t('Never'),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('graphql_shield.api_keys');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Delete Permanently');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->authManager->deleteApiKey($this->keyId);

    $this->messenger()->addStatus($this->t('API key "@name" has been permanently deleted.', [
      '@name' => $this->keyData['name'] ?? 'Unknown',
    ]));

    $form_state->setRedirect('graphql_shield.api_keys');
  }

}
