<?php

namespace Drupal\graphql_shield\Form;

use Drupal\Core\Url;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\user\Entity\User;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\graphql_shield\Service\AuthenticationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to edit an existing API key configuration.
 */
class ApiKeyEditForm extends FormBase {

  /**
   * The authentication manager service.
   *
   * @var \Drupal\graphql_shield\Service\AuthenticationManager
   */
  protected $authManager;

  /**
   * Constructs an ApiKeyEditForm object.
   *
   * @param \Drupal\graphql_shield\Service\AuthenticationManager $auth_manager
   *   The authentication manager service.
   */
  public function __construct(AuthenticationManager $auth_manager) {
    $this->authManager = $auth_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.auth_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'graphql_shield_api_key_edit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    // Load the API key.
    $key_data = $this->authManager->getApiKey($id);

    if (!$key_data) {
      $this->messenger()->addError($this->t('API key not found.'));
      return $this->redirect('graphql_shield.api_keys');
    }

    // Store the ID for submission.
    $form_state->set('key_id', $id);

    $form['info'] = [
      '#markup' => '<p>' . $this->t('Edit API key configuration. The key itself cannot be changed for security reasons.') . '</p>',
    ];

    $form['key_prefix'] = [
      '#type' => 'item',
      '#title' => $this->t('Key Prefix'),
      '#markup' => '<code>' . $key_data['key_prefix'] . '...</code>',
    ];

    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Key Name'),
      '#description' => $this->t('A descriptive name for this API key.'),
      '#required' => TRUE,
      '#maxlength' => 255,
      '#default_value' => $key_data['name'],
    ];

    $form['uid'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => 'user',
      '#title' => $this->t('Associated User'),
      '#description' => $this->t('The user that will be associated with this API key.'),
      '#required' => TRUE,
      '#default_value' => User::load($key_data['uid']),
      '#selection_settings' => [
        'include_anonymous' => FALSE,
      ],
    ];

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#description' => $this->t('Uncheck to temporarily disable this API key without deleting it.'),
      '#default_value' => $key_data['enabled'],
    ];

    $form['rate_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Custom Rate Limit'),
      '#description' => $this->t('Optional: Set a custom rate limit for this key (requests per minute). Leave empty to use default.'),
      '#min' => 1,
      '#default_value' => $key_data['rate_limit'],
    ];

    $expires_value = NULL;
    if (!empty($key_data['expires'])) {
      $expires_value = DrupalDateTime::createFromTimestamp($key_data['expires']);
    }

    $form['expires'] = [
      '#type' => 'datetime',
      '#title' => $this->t('Expiration Date'),
      '#description' => $this->t('Optional: Set when this key should expire. Leave empty for no expiration.'),
      '#default_value' => $expires_value,
    ];

    $ip_whitelist_value = '';
    if (!empty($key_data['ip_whitelist'])) {
      $ips = unserialize($key_data['ip_whitelist']);
      if (is_array($ips)) {
        $ip_whitelist_value = implode("\n", $ips);
      }
    }

    $form['ip_whitelist'] = [
      '#type' => 'textarea',
      '#title' => $this->t('IP Whitelist'),
      '#description' => $this->t('Optional: Restrict this key to specific IP addresses (one per line). Leave empty to allow all IPs.'),
      '#rows' => 5,
      '#default_value' => $ip_whitelist_value,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save Changes'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('graphql_shield.api_keys'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $id = $form_state->get('key_id');
    $name = $form_state->getValue('name');

    // Get the selected user.
    $user_entity = $form_state->getValue('uid');
    $uid = $user_entity instanceof User ? $user_entity->id() : $user_entity;

    $enabled = $form_state->getValue('enabled') ? 1 : 0;

    // Prepare update data.
    $update_data = [
      'name' => $name,
      'uid' => $uid,
      'enabled' => $enabled,
    ];

    // Rate limit.
    $rate_limit = $form_state->getValue('rate_limit');
    $update_data['rate_limit'] = $rate_limit ?: NULL;

    // Expiration.
    $expires = $form_state->getValue('expires');
    if ($expires instanceof DrupalDateTime) {
      $update_data['expires'] = $expires->getTimestamp();
    }
    else {
      $update_data['expires'] = NULL;
    }

    // IP whitelist.
    $ip_whitelist = $form_state->getValue('ip_whitelist');
    if (!empty($ip_whitelist)) {
      $ips = array_filter(array_map('trim', explode("\n", $ip_whitelist)));
      $update_data['ip_whitelist'] = !empty($ips) ? serialize($ips) : NULL;
    }
    else {
      $update_data['ip_whitelist'] = NULL;
    }

    // Update the API key.
    try {
      \Drupal::database()->update('graphql_shield_api_keys')
        ->fields($update_data)
        ->condition('id', $id)
        ->execute();

      $this->messenger()->addStatus($this->t('API key "@name" has been updated.', ['@name' => $name]));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to update API key: @error', ['@error' => $e->getMessage()]));
    }

    // Redirect back to the list.
    $form_state->setRedirect('graphql_shield.api_keys');
  }

}
