<?php

namespace Drupal\graphql_shield\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\graphql_shield\Service\AuthenticationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for revoking an API key.
 */
class ApiKeyRevokeForm extends ConfirmFormBase {

  /**
   * The authentication manager.
   *
   * @var \Drupal\graphql_shield\Service\AuthenticationManager
   */
  protected $authManager;

  /**
   * The API key ID to revoke.
   *
   * @var int
   */
  protected $keyId;

  /**
   * The API key data.
   *
   * @var array
   */
  protected $keyData;

  /**
   * Constructs an ApiKeyRevokeForm object.
   */
  public function __construct(AuthenticationManager $auth_manager) {
    $this->authManager = $auth_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.auth_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'graphql_shield_api_key_revoke';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $id = NULL) {
    $this->keyId = $id;

    // Load the API key data.
    $this->keyData = $this->authManager->getApiKey($id);

    if (!$this->keyData) {
      $this->messenger()->addError($this->t('API key not found.'));
      return $this->redirect('graphql_shield.api_keys');
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to revoke the API key "@name"?', [
      '@name' => $this->keyData['name'] ?? 'Unknown',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This will disable the API key. Requests using this key will be rejected. The key can be deleted later if needed. Key prefix: @prefix', [
      '@prefix' => $this->keyData['key_prefix'] ?? 'N/A',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('graphql_shield.api_keys');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Revoke API Key');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->authManager->revokeApiKey($this->keyId);

    $this->messenger()->addStatus($this->t('API key "@name" has been revoked.', [
      '@name' => $this->keyData['name'] ?? 'Unknown',
    ]));

    $form_state->setRedirect('graphql_shield.api_keys');
  }

}
