<?php

namespace Drupal\graphql_shield\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\graphql_shield\Service\SecurityLogger;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for clearing security logs.
 */
class ClearLogsForm extends ConfirmFormBase {

  /**
   * The security logger service.
   *
   * @var \Drupal\graphql_shield\Service\SecurityLogger
   */
  protected $securityLogger;

  /**
   * Constructs a ClearLogsForm object.
   */
  public function __construct(SecurityLogger $security_logger) {
    $this->securityLogger = $security_logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.security_logger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'graphql_shield_clear_logs';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Are you sure you want to clear all security logs?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This will permanently delete all GraphQL Shield security logs. This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('graphql_shield.dashboard');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Clear All Logs');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $deleted = $this->securityLogger->clearAllLogs();

    $this->messenger()->addStatus($this->t('Successfully cleared @count security log entries.', [
      '@count' => $deleted,
    ]));

    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
