<?php

namespace Drupal\graphql_shield\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure GraphQL Shield settings.
 */
class GraphQLShieldSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['graphql_shield.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'graphql_shield_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('graphql_shield.settings');

    $form['tabs'] = [
      '#type' => 'vertical_tabs',
    ];

    // Query Complexity Settings.
    $form['complexity'] = [
      '#type' => 'details',
      '#title' => $this->t('Query Complexity & Depth'),
      '#group' => 'tabs',
    ];

    $form['complexity']['query_complexity_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable query complexity analysis'),
      '#default_value' => $config->get('query_complexity.enabled'),
    ];

    $form['complexity']['query_complexity_max_depth'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum query depth'),
      '#default_value' => $config->get('query_complexity.max_depth') ?: 10,
      '#min' => 1,
      '#max' => 50,
    ];

    $form['complexity']['query_complexity_max_complexity'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum query complexity score'),
      '#default_value' => $config->get('query_complexity.max_complexity') ?: 1000,
      '#min' => 100,
    ];

    // Rate Limiting Settings.
    $form['rate_limiting'] = [
      '#type' => 'details',
      '#title' => $this->t('Rate Limiting'),
      '#group' => 'tabs',
    ];

    $form['rate_limiting']['rate_limiting_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable rate limiting'),
      '#default_value' => $config->get('rate_limiting.enabled'),
    ];

    $form['rate_limiting']['rate_limiting_per_user'] = [
      '#type' => 'number',
      '#title' => $this->t('Requests per user'),
      '#default_value' => $config->get('rate_limiting.per_user') ?: 100,
      '#description' => $this->t('Maximum requests per authenticated user per time window.'),
    ];

    $form['rate_limiting']['rate_limiting_per_ip'] = [
      '#type' => 'number',
      '#title' => $this->t('Requests per IP'),
      '#default_value' => $config->get('rate_limiting.per_ip') ?: 60,
      '#description' => $this->t('Maximum requests per IP address per time window.'),
    ];

    $form['rate_limiting']['rate_limiting_time_window'] = [
      '#type' => 'number',
      '#title' => $this->t('Time window (seconds)'),
      '#default_value' => $config->get('rate_limiting.time_window') ?: 60,
    ];

    // Persisted Queries Settings.
    $form['persisted'] = [
      '#type' => 'details',
      '#title' => $this->t('Persisted Queries'),
      '#group' => 'tabs',
    ];

    $form['persisted']['persisted_queries_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable persisted queries'),
      '#default_value' => $config->get('persisted_queries.enabled'),
    ];

    $form['persisted']['persisted_queries_development_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Development mode'),
      '#default_value' => $config->get('persisted_queries.development_mode'),
      '#description' => $this->t('Allow non-persisted queries (disable in production).'),
    ];

    // Introspection Settings.
    $form['introspection'] = [
      '#type' => 'details',
      '#title' => $this->t('Introspection Control'),
      '#group' => 'tabs',
    ];

    $form['introspection']['introspection_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable introspection'),
      '#default_value' => $config->get('introspection.enabled'),
    ];

    $form['introspection']['introspection_allow_authenticated'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow introspection for authenticated users'),
      '#default_value' => $config->get('introspection.allow_authenticated'),
    ];

    // Request Limits Settings.
    $form['request_limits'] = [
      '#type' => 'details',
      '#title' => $this->t('Request Limits'),
      '#group' => 'tabs',
    ];

    $form['request_limits']['request_limits_max_query_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum query size (bytes)'),
      '#default_value' => $config->get('request_limits.max_query_size') ?: 10000,
    ];

    $form['request_limits']['request_limits_max_variables'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum number of variables'),
      '#default_value' => $config->get('request_limits.max_variables') ?: 50,
    ];

    $form['request_limits']['request_limits_max_batch_queries'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum batch queries'),
      '#default_value' => $config->get('request_limits.max_batch_queries') ?: 10,
    ];

    $form['request_limits']['request_limits_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Query timeout (seconds)'),
      '#default_value' => $config->get('request_limits.timeout') ?: 30,
    ];

    // IP Restrictions.
    $form['ip_restrictions'] = [
      '#type' => 'details',
      '#title' => $this->t('IP Restrictions'),
      '#group' => 'tabs',
    ];

    $form['ip_restrictions']['ip_restrictions_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable IP restrictions'),
      '#default_value' => $config->get('ip_restrictions.enabled'),
    ];

    $form['ip_restrictions']['ip_restrictions_auto_block_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable automatic IP blocking'),
      '#default_value' => $config->get('ip_restrictions.auto_block_enabled'),
    ];

    $form['ip_restrictions']['ip_restrictions_auto_block_duration'] = [
      '#type' => 'number',
      '#title' => $this->t('Auto-block duration (seconds)'),
      '#default_value' => $config->get('ip_restrictions.auto_block_duration') ?: 3600,
    ];

    // DoS Protection.
    $form['dos_protection'] = [
      '#type' => 'details',
      '#title' => $this->t('DoS/DDoS Protection'),
      '#group' => 'tabs',
    ];

    $form['dos_protection']['dos_protection_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable DoS protection'),
      '#default_value' => $config->get('dos_protection.enabled'),
    ];

    $form['dos_protection']['dos_protection_circuit_breaker_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable circuit breaker'),
      '#default_value' => $config->get('dos_protection.circuit_breaker_enabled'),
    ];

    $form['dos_protection']['dos_protection_max_aliases'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum aliases per query'),
      '#default_value' => $config->get('dos_protection.max_aliases') ?: 50,
    ];

    // Logging Settings.
    $form['logging'] = [
      '#type' => 'details',
      '#title' => $this->t('Logging & Monitoring'),
      '#group' => 'tabs',
    ];

    $form['logging']['logging_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable security logging'),
      '#default_value' => $config->get('logging.enabled'),
    ];

    $form['logging']['logging_log_all_queries'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log all queries'),
      '#default_value' => $config->get('logging.log_all_queries'),
      '#description' => $this->t('Warning: May impact performance.'),
    ];

    $form['logging']['logging_log_slow_queries'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log slow queries'),
      '#default_value' => $config->get('logging.log_slow_queries'),
    ];

    $form['logging']['logging_slow_query_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Slow query threshold (seconds)'),
      '#default_value' => $config->get('logging.slow_query_threshold') ?: 5,
    ];

    // Authentication Settings.
    $form['authentication'] = [
      '#type' => 'details',
      '#title' => $this->t('Authentication & API Keys'),
      '#group' => 'tabs',
    ];

    $form['authentication']['auth_require_api_key'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Require API key for all requests'),
      '#default_value' => $config->get('auth.require_api_key'),
      '#description' => $this->t('If enabled, all GraphQL requests must include a valid API key in the X-API-Key header.'),
    ];

    $form['authentication']['auth_allow_authenticated_without_key'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow authenticated users without API key'),
      '#default_value' => $config->get('auth.allow_authenticated_without_key'),
      '#description' => $this->t('If enabled, logged-in Drupal users can make requests without an API key. Only applies when "Require API key" is enabled.'),
      '#states' => [
        'visible' => [
          ':input[name="auth_require_api_key"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['authentication']['help'] = [
      '#markup' => '<div class="messages messages--warning">' .
      $this->t('<strong>Note:</strong> When API key requirement is enabled, external applications will need to include <code>X-API-Key: your-api-key</code> header in all requests. Generate API keys at <a href="/admin/config/graphql/shield/api-keys">API Keys Management</a>.') .
      '</div>',
    ];

    // CORS Settings.
    $form['cors'] = [
      '#type' => 'details',
      '#title' => $this->t('CORS Configuration'),
      '#group' => 'tabs',
    ];

    $form['cors']['cors_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable CORS management'),
      '#default_value' => $config->get('cors.enabled'),
    ];

    $form['cors']['cors_allowed_origins'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed origins'),
      '#default_value' => implode("\n", $config->get('cors.allowed_origins') ?: ['*']),
      '#description' => $this->t('One origin per line. Use * for all origins.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('graphql_shield.settings');

    // Query Complexity.
    $config->set('query_complexity.enabled', $form_state->getValue('query_complexity_enabled'));
    $config->set('query_complexity.max_depth', $form_state->getValue('query_complexity_max_depth'));
    $config->set('query_complexity.max_complexity', $form_state->getValue('query_complexity_max_complexity'));

    // Rate Limiting.
    $config->set('rate_limiting.enabled', $form_state->getValue('rate_limiting_enabled'));
    $config->set('rate_limiting.per_user', $form_state->getValue('rate_limiting_per_user'));
    $config->set('rate_limiting.per_ip', $form_state->getValue('rate_limiting_per_ip'));
    $config->set('rate_limiting.time_window', $form_state->getValue('rate_limiting_time_window'));

    // Persisted Queries.
    $config->set('persisted_queries.enabled', $form_state->getValue('persisted_queries_enabled'));
    $config->set('persisted_queries.development_mode', $form_state->getValue('persisted_queries_development_mode'));

    // Introspection.
    $config->set('introspection.enabled', $form_state->getValue('introspection_enabled'));
    $config->set('introspection.allow_authenticated', $form_state->getValue('introspection_allow_authenticated'));

    // Request Limits.
    $config->set('request_limits.max_query_size', $form_state->getValue('request_limits_max_query_size'));
    $config->set('request_limits.max_variables', $form_state->getValue('request_limits_max_variables'));
    $config->set('request_limits.max_batch_queries', $form_state->getValue('request_limits_max_batch_queries'));
    $config->set('request_limits.timeout', $form_state->getValue('request_limits_timeout'));

    // IP Restrictions.
    $config->set('ip_restrictions.enabled', $form_state->getValue('ip_restrictions_enabled'));
    $config->set('ip_restrictions.auto_block_enabled', $form_state->getValue('ip_restrictions_auto_block_enabled'));
    $config->set('ip_restrictions.auto_block_duration', $form_state->getValue('ip_restrictions_auto_block_duration'));

    // DoS Protection.
    $config->set('dos_protection.enabled', $form_state->getValue('dos_protection_enabled'));
    $config->set('dos_protection.circuit_breaker_enabled', $form_state->getValue('dos_protection_circuit_breaker_enabled'));
    $config->set('dos_protection.max_aliases', $form_state->getValue('dos_protection_max_aliases'));

    // Logging.
    $config->set('logging.enabled', $form_state->getValue('logging_enabled'));
    $config->set('logging.log_all_queries', $form_state->getValue('logging_log_all_queries'));
    $config->set('logging.log_slow_queries', $form_state->getValue('logging_log_slow_queries'));
    $config->set('logging.slow_query_threshold', $form_state->getValue('logging_slow_query_threshold'));

    // Authentication.
    $config->set('auth.require_api_key', $form_state->getValue('auth_require_api_key'));
    $config->set('auth.allow_authenticated_without_key', $form_state->getValue('auth_allow_authenticated_without_key'));

    // CORS.
    $config->set('cors.enabled', $form_state->getValue('cors_enabled'));
    $origins = array_filter(array_map('trim', explode("\n", $form_state->getValue('cors_allowed_origins'))));
    $config->set('cors.allowed_origins', $origins);

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
