<?php

namespace Drupal\graphql_shield\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\graphql_shield\Service\IpRestrictor;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for managing IP restrictions.
 */
class IpManagementForm extends FormBase {

  /**
   * The IP restrictor service.
   *
   * @var \Drupal\graphql_shield\Service\IpRestrictor
   */
  protected $ipRestrictor;

  /**
   * Constructs an IpManagementForm object.
   */
  public function __construct(IpRestrictor $ip_restrictor) {
    $this->ipRestrictor = $ip_restrictor;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.ip_restrictor')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'graphql_shield_ip_management';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['add'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Add IP Rule'),
    ];

    $form['add']['ip_address'] = [
      '#type' => 'textfield',
      '#title' => $this->t('IP Address'),
      '#required' => TRUE,
    ];

    $form['add']['rule_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Rule Type'),
      '#options' => [
        'allow' => $this->t('Allow (Whitelist)'),
        'block' => $this->t('Block (Blacklist)'),
      ],
      '#default_value' => 'block',
      '#required' => TRUE,
    ];

    $form['add']['reason'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Reason'),
      '#rows' => 3,
    ];

    $form['add']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Rule'),
    ];

    // List existing rules.
    $form['existing'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Existing IP Rules'),
    ];

    $blocked = $this->ipRestrictor->listRules('block');
    $allowed = $this->ipRestrictor->listRules('allow');

    if (!empty($blocked)) {
      $form['existing']['blocked'] = [
        '#type' => 'details',
        '#title' => $this->t('Blocked IPs (@count)', ['@count' => count($blocked)]),
        '#open' => TRUE,
      ];

      $rows = [];
      foreach ($blocked as $rule) {
        $rows[] = [
          $rule['ip_address'],
          $rule['reason'] ?? '',
          date('Y-m-d H:i:s', $rule['created']),
          $rule['auto_created'] ? $this->t('Auto') : $this->t('Manual'),
        ];
      }

      $form['existing']['blocked']['table'] = [
        '#theme' => 'table',
        '#header' => [
          $this->t('IP Address'),
          $this->t('Reason'),
          $this->t('Created'),
          $this->t('Type'),
        ],
        '#rows' => $rows,
      ];
    }

    if (!empty($allowed)) {
      $form['existing']['allowed'] = [
        '#type' => 'details',
        '#title' => $this->t('Allowed IPs (@count)', ['@count' => count($allowed)]),
        '#open' => FALSE,
      ];

      $rows = [];
      foreach ($allowed as $rule) {
        $rows[] = [
          $rule['ip_address'],
          $rule['reason'] ?? '',
          date('Y-m-d H:i:s', $rule['created']),
        ];
      }

      $form['existing']['allowed']['table'] = [
        '#theme' => 'table',
        '#header' => [
          $this->t('IP Address'),
          $this->t('Reason'),
          $this->t('Created'),
        ],
        '#rows' => $rows,
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $ip = $form_state->getValue('ip_address');
    $type = $form_state->getValue('rule_type');
    $reason = $form_state->getValue('reason');

    if ($type === 'block') {
      $this->ipRestrictor->blockIp($ip, $reason);
      $this->messenger()->addStatus($this->t('IP address @ip has been blocked.', ['@ip' => $ip]));
    }
    else {
      $this->ipRestrictor->allowIp($ip, $reason);
      $this->messenger()->addStatus($this->t('IP address @ip has been added to allowlist.', ['@ip' => $ip]));
    }
  }

}
