<?php

namespace Drupal\graphql_shield\Form;

use Drupal\Core\Url;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\graphql_shield\Service\PersistedQueryManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for adding a new persisted query.
 */
class PersistedQueryAddForm extends FormBase {

  /**
   * The persisted query manager.
   *
   * @var \Drupal\graphql_shield\Service\PersistedQueryManager
   */
  protected $persistedQueryManager;

  /**
   * Constructs a PersistedQueryAddForm object.
   */
  public function __construct(PersistedQueryManager $persisted_query_manager) {
    $this->persistedQueryManager = $persisted_query_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('graphql_shield.persisted_query')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'graphql_shield_persisted_query_add';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['query_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Query ID'),
      '#description' => $this->t('A unique identifier for this query (e.g., "GetArticles", "UserProfile").'),
      '#required' => TRUE,
      '#maxlength' => 255,
    ];

    $form['query'] = [
      '#type' => 'textarea',
      '#title' => $this->t('GraphQL Query'),
      '#description' => $this->t('The complete GraphQL query.'),
      '#required' => TRUE,
      '#rows' => 15,
      '#attributes' => [
        'style' => 'font-family: monospace;',
      ],
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#description' => $this->t('Optional: Describe what this query does.'),
      '#rows' => 3,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add Persisted Query'),
      '#button_type' => 'primary',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => Url::fromRoute('graphql_shield.persisted_queries'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $query_id = $form_state->getValue('query_id');
    $query = $form_state->getValue('query');

    // Validate query ID is alphanumeric with underscores.
    if (!preg_match('/^[a-zA-Z0-9_]+$/', $query_id)) {
      $form_state->setErrorByName('query_id', $this->t('Query ID must contain only letters, numbers, and underscores.'));
    }

    // Validate query is not empty.
    if (empty(trim($query))) {
      $form_state->setErrorByName('query', $this->t('Query cannot be empty.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $query_id = $form_state->getValue('query_id');
    $query = $form_state->getValue('query');
    $description = $form_state->getValue('description');

    try {
      $this->persistedQueryManager->addQuery($query_id, $query, $description);
      $this->messenger()->addStatus($this->t('Persisted query "@id" has been added.', ['@id' => $query_id]));
      $form_state->setRedirect('graphql_shield.persisted_queries');
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error adding persisted query: @error', ['@error' => $e->getMessage()]));
    }
  }

}
