<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Database\Connection;
use Psr\Log\LoggerInterface;

/**
 * Service for GraphQL authentication and authorization.
 */
class AuthenticationManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an AuthenticationManager object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountProxyInterface $current_user,
    Connection $database,
    LoggerInterface $logger,
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->database = $database;
    $this->logger = $logger;
  }

  /**
   * Validates API key from request.
   *
   * @param string $api_key
   *   The API key from request header.
   *
   * @return array
   *   Array with 'valid' boolean and 'key_data' if valid.
   */
  public function validateApiKey($api_key) {
    if (empty($api_key)) {
      return ['valid' => FALSE];
    }

    $key_hash = hash('sha256', $api_key);

    $key_data = $this->database->select('graphql_shield_api_keys', 'k')
      ->fields('k')
      ->condition('key_hash', $key_hash)
      ->condition('enabled', 1)
      ->execute()
      ->fetchAssoc();

    if (!$key_data) {
      $this->logger->warning('Invalid API key attempt');
      return ['valid' => FALSE];
    }

    // Check expiration.
    if ($key_data['expires'] && $key_data['expires'] < time()) {
      $this->logger->warning('Expired API key used: @name', ['@name' => $key_data['name']]);
      return ['valid' => FALSE, 'error' => 'API key has expired'];
    }

    // Update last used timestamp.
    $this->database->update('graphql_shield_api_keys')
      ->fields([
        'last_used' => time(),
      ])
      ->expression('usage_count', 'usage_count + 1')
      ->condition('id', $key_data['id'])
      ->execute();

    return ['valid' => TRUE, 'key_data' => $key_data];
  }

  /**
   * Validates JWT token.
   *
   * @param string $token
   *   JWT token.
   *
   * @return array
   *   Validation result.
   */
  public function validateJwtToken($token) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('jwt.enabled')) {
      return ['valid' => FALSE, 'error' => 'JWT authentication is disabled'];
    }

    // Basic JWT validation (in production, use a proper JWT library).
    $parts = explode('.', $token);

    if (count($parts) !== 3) {
      return ['valid' => FALSE, 'error' => 'Invalid JWT format'];
    }

    [$header, $payload, $signature] = $parts;

    try {
      $payload_data = json_decode(base64_decode($payload), TRUE);

      // Check expiration.
      if (isset($payload_data['exp']) && $payload_data['exp'] < time()) {
        return ['valid' => FALSE, 'error' => 'Token has expired'];
      }

      // Verify signature (simplified - use proper JWT library in production).
      $secret = $config->get('jwt.secret');
      $expected_signature = hash_hmac('sha256', "$header.$payload", $secret);

      if (!hash_equals($expected_signature, base64_decode($signature))) {
        $this->logger->warning('Invalid JWT signature');
        return ['valid' => FALSE, 'error' => 'Invalid signature'];
      }

      return ['valid' => TRUE, 'payload' => $payload_data];
    }
    catch (\Exception $e) {
      $this->logger->error('JWT validation error: @error', ['@error' => $e->getMessage()]);
      return ['valid' => FALSE, 'error' => 'Token validation failed'];
    }
  }

  /**
   * Checks if user has permission for GraphQL operation.
   *
   * @param string $operation
   *   Operation type (query, mutation, subscription).
   * @param array $context
   *   Additional context.
   *
   * @return bool
   *   TRUE if authorized.
   */
  public function isAuthorized($operation, array $context = []) {
    // Bypass permission for admin users.
    if ($this->currentUser->hasPermission('bypass graphql shield')) {
      return TRUE;
    }

    $config = $this->configFactory->get('graphql_shield.settings');

    // Check if authentication is required.
    if ($config->get('auth.require_authentication')) {
      if ($this->currentUser->isAnonymous()) {
        return FALSE;
      }
    }

    // Check operation-specific permissions.
    $permission_map = [
      'query' => 'execute graphql queries',
      'mutation' => 'execute graphql mutations',
      'subscription' => 'execute graphql subscriptions',
    ];

    if (isset($permission_map[$operation])) {
      return $this->currentUser->hasPermission($permission_map[$operation]);
    }

    return TRUE;
  }

  /**
   * Generates a new API key.
   *
   * @param string $name
   *   Key name/label.
   * @param int $uid
   *   Owner user ID.
   * @param array $options
   *   Additional options (roles, rate_limit, ip_whitelist, expires).
   *
   * @return array
   *   Array with 'key' (plaintext, show once) and 'id'.
   */
  public function generateApiKey($name, $uid, array $options = []) {
    // Generate random key (32 bytes = 64 hex chars).
    $key = bin2hex(random_bytes(32));
    $key_hash = hash('sha256', $key);
    $key_prefix = substr($key, 0, 8);

    $id = $this->database->insert('graphql_shield_api_keys')
      ->fields([
        'key_hash' => $key_hash,
        'key_prefix' => $key_prefix,
        'name' => $name,
        'uid' => $uid,
        'roles' => isset($options['roles']) ? serialize($options['roles']) : NULL,
        'rate_limit' => $options['rate_limit'] ?? NULL,
        'ip_whitelist' => isset($options['ip_whitelist']) ? serialize($options['ip_whitelist']) : NULL,
        'enabled' => 1,
        'created' => time(),
        'expires' => $options['expires'] ?? NULL,
        'last_used' => NULL,
        'usage_count' => 0,
      ])
      ->execute();

    $this->logger->info('API key created: @name', ['@name' => $name]);

    return ['key' => $key, 'id' => $id, 'prefix' => $key_prefix];
  }

  /**
   * Revokes an API key.
   *
   * @param int $id
   *   Key ID.
   */
  public function revokeApiKey($id) {
    $this->database->update('graphql_shield_api_keys')
      ->fields(['enabled' => 0])
      ->condition('id', $id)
      ->execute();

    $this->logger->info('API key revoked: ID @id', ['@id' => $id]);
  }

  /**
   * Deletes an API key permanently.
   *
   * @param int $id
   *   Key ID.
   */
  public function deleteApiKey($id) {
    $this->database->delete('graphql_shield_api_keys')
      ->condition('id', $id)
      ->execute();

    $this->logger->info('API key deleted: ID @id', ['@id' => $id]);
  }

  /**
   * Gets a single API key by ID.
   *
   * @param int $id
   *   Key ID.
   *
   * @return array|null
   *   API key data or NULL if not found.
   */
  public function getApiKey($id) {
    return $this->database->select('graphql_shield_api_keys', 'k')
      ->fields('k')
      ->condition('id', $id)
      ->execute()
      ->fetchAssoc();
  }

  /**
   * Lists API keys for a user.
   *
   * @param int $uid
   *   User ID.
   *
   * @return array
   *   Array of API keys.
   */
  public function listApiKeys($uid = NULL) {
    $query = $this->database->select('graphql_shield_api_keys', 'k')
      ->fields('k');

    if ($uid !== NULL) {
      $query->condition('uid', $uid);
    }

    return $query->orderBy('created', 'DESC')
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);
  }

}
