<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for managing CORS headers.
 */
class CorsManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a CorsManager object.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Gets CORS headers for GraphQL requests.
   *
   * @param string $origin
   *   Request origin.
   *
   * @return array
   *   Array of headers to set.
   */
  public function getHeaders($origin) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('cors.enabled')) {
      return [];
    }

    $headers = [];

    // Check if origin is allowed.
    if ($this->isOriginAllowed($origin)) {
      $headers['Access-Control-Allow-Origin'] = $origin;
    }
    else {
      $allowed_origins = $config->get('cors.allowed_origins') ?: ['*'];
      if (in_array('*', $allowed_origins)) {
        $headers['Access-Control-Allow-Origin'] = '*';
      }
    }

    // Allowed methods.
    $allowed_methods = $config->get('cors.allowed_methods') ?: ['GET', 'POST', 'OPTIONS'];
    $headers['Access-Control-Allow-Methods'] = implode(', ', $allowed_methods);

    // Allowed headers.
    $allowed_headers = $config->get('cors.allowed_headers') ?: [
      'Content-Type',
      'Authorization',
      'X-API-Key',
    ];
    $headers['Access-Control-Allow-Headers'] = implode(', ', $allowed_headers);

    // Credentials.
    if ($config->get('cors.allow_credentials')) {
      $headers['Access-Control-Allow-Credentials'] = 'true';
    }

    // Max age.
    $max_age = $config->get('cors.max_age') ?: 86400;
    $headers['Access-Control-Max-Age'] = (string) $max_age;

    // Exposed headers.
    $exposed_headers = $config->get('cors.exposed_headers') ?: [
      'X-RateLimit-Limit',
      'X-RateLimit-Remaining',
      'X-RateLimit-Reset',
    ];
    $headers['Access-Control-Expose-Headers'] = implode(', ', $exposed_headers);

    return $headers;
  }

  /**
   * Checks if origin is allowed.
   *
   * @param string $origin
   *   Request origin.
   *
   * @return bool
   *   TRUE if allowed.
   */
  protected function isOriginAllowed($origin) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $allowed_origins = $config->get('cors.allowed_origins') ?: ['*'];

    if (in_array('*', $allowed_origins)) {
      return TRUE;
    }

    // Check exact match.
    if (in_array($origin, $allowed_origins)) {
      return TRUE;
    }

    // Check pattern match (convert wildcard to regex).
    foreach ($allowed_origins as $allowed) {
      // Convert shell wildcard pattern to regex pattern.
      $pattern = '/^' . str_replace(
        ['\*', '\?'],
        ['.*', '.'],
        preg_quote($allowed, '/')
      ) . '$/i';
      if (preg_match($pattern, $origin)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Gets security headers.
   *
   * @return array
   *   Security headers.
   */
  public function getSecurityHeaders() {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('security_headers.enabled')) {
      return [];
    }

    $headers = [];

    // Content Security Policy.
    if ($csp = $config->get('security_headers.csp')) {
      $headers['Content-Security-Policy'] = $csp;
    }

    // X-Frame-Options.
    $headers['X-Frame-Options'] = $config->get('security_headers.x_frame_options') ?: 'DENY';

    // X-Content-Type-Options.
    $headers['X-Content-Type-Options'] = 'nosniff';

    // X-XSS-Protection.
    $headers['X-XSS-Protection'] = '1; mode=block';

    // Referrer-Policy.
    $headers['Referrer-Policy'] = $config->get('security_headers.referrer_policy') ?: 'no-referrer';

    // Strict-Transport-Security.
    if ($config->get('security_headers.hsts_enabled')) {
      $max_age = $config->get('security_headers.hsts_max_age') ?: 31536000;
      $headers['Strict-Transport-Security'] = "max-age=$max_age; includeSubDomains";
    }

    return $headers;
  }

}
