<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Cache\CacheBackendInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for DoS/DDoS protection.
 */
class DosProtector {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a DosProtector object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    CacheBackendInterface $cache,
    RequestStack $request_stack,
  ) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->cache = $cache;
    $this->requestStack = $request_stack;
  }

  /**
   * Checks for DoS/DDoS patterns.
   *
   * @param string $query
   *   The GraphQL query.
   *
   * @return array
   *   Array with 'threat_detected' boolean and optional 'reason'.
   */
  public function checkThreat($query) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('dos_protection.enabled')) {
      return ['threat_detected' => FALSE];
    }

    $request = $this->requestStack->getCurrentRequest();
    $ip = $request ? $request->getClientIp() : '0.0.0.0';

    // Check for query deduplication (identical queries in short time).
    if ($this->detectQueryFlooding($query, $ip)) {
      return ['threat_detected' => TRUE, 'reason' => 'Query flooding detected'];
    }

    // Check for connection flooding.
    if ($this->detectConnectionFlooding($ip)) {
      return ['threat_detected' => TRUE, 'reason' => 'Too many concurrent connections'];
    }

    // Check for alias flooding.
    if ($this->detectAliasFlooding($query)) {
      return ['threat_detected' => TRUE, 'reason' => 'Alias flooding detected'];
    }

    return ['threat_detected' => FALSE];
  }

  /**
   * Detects query flooding (same query repeated rapidly).
   *
   * @param string $query
   *   The query.
   * @param string $ip
   *   IP address.
   *
   * @return bool
   *   TRUE if flooding detected.
   */
  protected function detectQueryFlooding($query, $ip) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $threshold = $config->get('dos_protection.duplicate_query_threshold') ?: 10;
    $time_window = $config->get('dos_protection.duplicate_query_window') ?: 60;

    $query_hash = hash('md5', $query);
    $cache_key = "graphql_shield:query_flood:$ip:$query_hash";

    $cached = $this->cache->get($cache_key);

    if (!$cached) {
      $this->cache->set($cache_key, 1, time() + $time_window);
      return FALSE;
    }

    $count = $cached->data + 1;
    $this->cache->set($cache_key, $count, time() + $time_window);

    return $count > $threshold;
  }

  /**
   * Detects connection flooding.
   *
   * @param string $ip
   *   IP address.
   *
   * @return bool
   *   TRUE if flooding detected.
   */
  protected function detectConnectionFlooding($ip) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $max_connections = $config->get('dos_protection.max_concurrent_connections') ?: 10;

    $cache_key = "graphql_shield:connections:$ip";
    $cached = $this->cache->get($cache_key);

    $current_connections = $cached ? $cached->data : 0;

    return $current_connections >= $max_connections;
  }

  /**
   * Detects alias flooding in query.
   *
   * @param string $query
   *   The query.
   *
   * @return bool
   *   TRUE if flooding detected.
   */
  protected function detectAliasFlooding($query) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $max_aliases = $config->get('dos_protection.max_aliases') ?: 50;

    // Count aliases (pattern: word: word).
    $alias_count = preg_match_all('/\b\w+\s*:\s*\w+/', $query);

    return $alias_count > $max_aliases;
  }

  /**
   * Increments connection counter.
   *
   * @param string|null $ip
   *   IP address.
   */
  public function incrementConnections($ip = NULL) {
    if ($ip === NULL) {
      $request = $this->requestStack->getCurrentRequest();
      $ip = $request ? $request->getClientIp() : '0.0.0.0';
    }

    $cache_key = "graphql_shield:connections:$ip";
    $cached = $this->cache->get($cache_key);

    $count = $cached ? $cached->data + 1 : 1;
    $this->cache->set($cache_key, $count, time() + 60);
  }

  /**
   * Decrements connection counter.
   *
   * @param string|null $ip
   *   IP address.
   */
  public function decrementConnections($ip = NULL) {
    if ($ip === NULL) {
      $request = $this->requestStack->getCurrentRequest();
      $ip = $request ? $request->getClientIp() : '0.0.0.0';
    }

    $cache_key = "graphql_shield:connections:$ip";
    $cached = $this->cache->get($cache_key);

    if ($cached && $cached->data > 0) {
      $count = $cached->data - 1;
      $this->cache->set($cache_key, $count, time() + 60);
    }
  }

  /**
   * Checks circuit breaker status.
   *
   * @return array
   *   Status with 'open' boolean.
   */
  public function checkCircuitBreaker() {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('dos_protection.circuit_breaker_enabled')) {
      return ['open' => FALSE];
    }

    $cache_key = 'graphql_shield:circuit_breaker';
    $cached = $this->cache->get($cache_key);

    if ($cached && $cached->data['open']) {
      $reset_time = $cached->data['reset_time'];

      if (time() < $reset_time) {
        return ['open' => TRUE, 'reset_time' => $reset_time];
      }

      // Reset circuit breaker.
      $this->cache->delete($cache_key);
      return ['open' => FALSE];
    }

    return ['open' => FALSE];
  }

  /**
   * Trips the circuit breaker.
   *
   * @param string $reason
   *   Reason for tripping.
   */
  public function tripCircuitBreaker($reason) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $duration = $config->get('dos_protection.circuit_breaker_duration') ?: 300;

    $cache_key = 'graphql_shield:circuit_breaker';
    $this->cache->set($cache_key, [
      'open' => TRUE,
      'reset_time' => time() + $duration,
      'reason' => $reason,
    ], time() + $duration);
  }

  /**
   * Resets the circuit breaker manually.
   */
  public function resetCircuitBreaker() {
    $this->cache->delete('graphql_shield:circuit_breaker');
  }

}
