<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for handling GraphQL errors securely.
 */
class ErrorHandler {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an ErrorHandler object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerInterface $logger) {
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * Sanitizes error messages for client response.
   *
   * @param array $errors
   *   Array of errors.
   *
   * @return array
   *   Sanitized errors.
   */
  public function sanitizeErrors(array $errors) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('error_handling.sanitize')) {
      return $errors;
    }

    $sanitized = [];

    foreach ($errors as $error) {
      $sanitized[] = $this->sanitizeError($error);
    }

    return $sanitized;
  }

  /**
   * Sanitizes a single error.
   *
   * @param array $error
   *   Error array.
   *
   * @return array
   *   Sanitized error.
   */
  protected function sanitizeError(array $error) {
    $config = $this->configFactory->get('graphql_shield.settings');

    // Log full error for debugging.
    $this->logger->error('GraphQL error: @message', [
      '@message' => $error['message'] ?? 'Unknown error',
    ]);

    // Remove sensitive information.
    $sanitized = [
      'message' => $this->sanitizeMessage($error['message'] ?? 'An error occurred'),
    ];

    // Remove stack traces in production.
    if ($config->get('error_handling.hide_stack_traces')) {
      unset($error['trace']);
      unset($error['debugMessage']);
    }

    // Keep error locations if configured.
    if ($config->get('error_handling.include_locations') && isset($error['locations'])) {
      $sanitized['locations'] = $error['locations'];
    }

    // Keep error paths if configured.
    if ($config->get('error_handling.include_paths') && isset($error['path'])) {
      $sanitized['path'] = $error['path'];
    }

    // Add custom error code.
    if (isset($error['extensions']['code'])) {
      $sanitized['extensions']['code'] = $error['extensions']['code'];
    }
    else {
      $sanitized['extensions']['code'] = 'INTERNAL_ERROR';
    }

    return $sanitized;
  }

  /**
   * Sanitizes error message.
   *
   * @param string $message
   *   Original message.
   *
   * @return string
   *   Sanitized message.
   */
  protected function sanitizeMessage($message) {
    $config = $this->configFactory->get('graphql_shield.settings');

    // Use generic messages in production.
    if ($config->get('error_handling.use_generic_messages')) {
      // Map specific errors to generic messages.
      $generic_map = [
        'validation' => 'Invalid query structure',
        'authentication' => 'Authentication failed',
        'authorization' => 'Access denied',
        'rate_limit' => 'Too many requests',
      ];

      foreach ($generic_map as $keyword => $generic) {
        if (stripos($message, $keyword) !== FALSE) {
          return $generic;
        }
      }

      return 'An error occurred processing your request';
    }

    // Remove file paths.
    $message = preg_replace('/\/[\w\/]+\.php/', '[removed]', $message);

    // Remove line numbers.
    $message = preg_replace('/on line \d+/', '', $message);

    // Remove database table names.
    $message = preg_replace('/\{[a-z_]+\}/', '[table]', $message);

    return $message;
  }

  /**
   * Creates a standardized error response.
   *
   * @param string $code
   *   Error code.
   * @param string $message
   *   Error message.
   * @param array $extra
   *   Additional data.
   *
   * @return array
   *   Error response.
   */
  public function createError($code, $message, array $extra = []) {
    $error = [
      'message' => $message,
      'extensions' => [
        'code' => $code,
      ],
    ];

    if (!empty($extra)) {
      $error['extensions'] = array_merge($error['extensions'], $extra);
    }

    return $error;
  }

  /**
   * Gets error code for security violations.
   *
   * @param string $violation_type
   *   Type of violation.
   *
   * @return string
   *   Error code.
   */
  public function getSecurityErrorCode($violation_type) {
    $codes = [
      'rate_limit' => 'RATE_LIMIT_EXCEEDED',
      'complexity' => 'QUERY_TOO_COMPLEX',
      'depth' => 'QUERY_TOO_DEEP',
      'size' => 'QUERY_TOO_LARGE',
      'ip_blocked' => 'IP_BLOCKED',
      'auth_failed' => 'AUTHENTICATION_FAILED',
      'access_denied' => 'ACCESS_DENIED',
      'invalid_token' => 'INVALID_TOKEN',
      'persisted_query' => 'PERSISTED_QUERY_NOT_FOUND',
      'introspection' => 'INTROSPECTION_DISABLED',
    ];

    return $codes[$violation_type] ?? 'SECURITY_VIOLATION';
  }

}
