<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for sanitizing GraphQL inputs.
 */
class InputSanitizer {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs an InputSanitizer object.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Sanitizes query variables.
   *
   * @param array $variables
   *   Query variables.
   *
   * @return array
   *   Sanitized variables.
   */
  public function sanitizeVariables(array $variables) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('input_validation.enabled')) {
      return $variables;
    }

    return $this->sanitizeArray($variables);
  }

  /**
   * Recursively sanitizes an array.
   *
   * @param array $data
   *   Data to sanitize.
   *
   * @return array
   *   Sanitized data.
   */
  protected function sanitizeArray(array $data) {
    $sanitized = [];

    foreach ($data as $key => $value) {
      // Sanitize key.
      $clean_key = $this->sanitizeString($key);

      // Sanitize value.
      if (is_array($value)) {
        $sanitized[$clean_key] = $this->sanitizeArray($value);
      }
      elseif (is_string($value)) {
        $sanitized[$clean_key] = $this->sanitizeString($value);
      }
      else {
        $sanitized[$clean_key] = $value;
      }
    }

    return $sanitized;
  }

  /**
   * Sanitizes a string value.
   *
   * @param string $value
   *   Value to sanitize.
   *
   * @return string
   *   Sanitized value.
   */
  protected function sanitizeString($value) {
    $config = $this->configFactory->get('graphql_shield.settings');

    // Remove null bytes.
    $value = str_replace("\0", '', $value);

    // Apply XSS filtering if enabled.
    if ($config->get('input_validation.xss_filter')) {
      $value = htmlspecialchars($value, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    }

    // Check for SQL injection patterns.
    if ($config->get('input_validation.sql_injection_check')) {
      $value = $this->checkSqlInjection($value);
    }

    return $value;
  }

  /**
   * Checks for SQL injection patterns.
   *
   * @param string $value
   *   Value to check.
   *
   * @return string
   *   Original value (throws exception if suspicious).
   */
  protected function checkSqlInjection($value) {
    $suspicious_patterns = [
      '/(\bUNION\b.*\bSELECT\b)/i',
      '/(\bDROP\b.*\bTABLE\b)/i',
      '/(\bINSERT\b.*\bINTO\b)/i',
      '/(\bUPDATE\b.*\bSET\b)/i',
      '/(\bDELETE\b.*\bFROM\b)/i',
      '/(\bEXEC\b.*\()/i',
      '/(\bSCRIPT\b.*>)/i',
    ];

    foreach ($suspicious_patterns as $pattern) {
      if (preg_match($pattern, $value)) {
        throw new \Exception('Potentially malicious input detected');
      }
    }

    return $value;
  }

  /**
   * Validates input types match schema.
   *
   * @param array $variables
   *   Variables.
   * @param array $schema
   *   Expected schema.
   *
   * @return bool
   *   TRUE if valid.
   */
  public function validateTypes(array $variables, array $schema) {
    foreach ($schema as $key => $expected_type) {
      if (!isset($variables[$key])) {
        continue;
      }

      $actual_type = gettype($variables[$key]);

      if (!$this->typesMatch($actual_type, $expected_type)) {
        return FALSE;
      }

      // Recursive validation for objects/arrays.
      if ($expected_type === 'object' && is_array($schema[$key])) {
        if (!$this->validateTypes($variables[$key], $schema[$key])) {
          return FALSE;
        }
      }
    }

    return TRUE;
  }

  /**
   * Checks if types match.
   *
   * @param string $actual
   *   Actual type.
   * @param string $expected
   *   Expected type.
   *
   * @return bool
   *   TRUE if match.
   */
  protected function typesMatch($actual, $expected) {
    $type_map = [
      'integer' => ['int', 'integer'],
      'double' => ['float', 'double'],
      'string' => ['string'],
      'boolean' => ['bool', 'boolean'],
      'array' => ['array', 'object'],
    ];

    foreach ($type_map as $php_type => $graphql_types) {
      if ($actual === $php_type && in_array($expected, $graphql_types)) {
        return TRUE;
      }
    }

    return FALSE;
  }

}
