<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Service for controlling GraphQL introspection queries.
 */
class IntrospectionController {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs an IntrospectionController object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, AccountProxyInterface $current_user) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
  }

  /**
   * Checks if introspection is allowed for current user.
   *
   * @return bool
   *   TRUE if allowed.
   */
  public function isAllowed() {
    $config = $this->configFactory->get('graphql_shield.settings');

    // Always allow for users with bypass permission.
    if ($this->currentUser->hasPermission('bypass graphql shield')) {
      return TRUE;
    }

    // Check if introspection is enabled globally.
    if (!$config->get('introspection.enabled')) {
      // Check if allowed for authenticated users only.
      if ($config->get('introspection.allow_authenticated') && !$this->currentUser->isAnonymous()) {
        return TRUE;
      }

      // Check role-based access.
      $allowed_roles = $config->get('introspection.allowed_roles') ?: [];
      if (!empty($allowed_roles)) {
        $user_roles = $this->currentUser->getRoles();
        if (array_intersect($user_roles, $allowed_roles)) {
          return TRUE;
        }
      }

      return FALSE;
    }

    return TRUE;
  }

  /**
   * Checks if a query contains introspection.
   *
   * @param string $query
   *   The GraphQL query.
   *
   * @return bool
   *   TRUE if query contains introspection.
   */
  public function containsIntrospection($query) {
    // Check for __schema or __type introspection queries.
    $introspection_patterns = [
      '/__schema\s*\{/',
      '/__type\s*\(/i',
      '/__typename/i',
      '/query\s+IntrospectionQuery/i',
    ];

    foreach ($introspection_patterns as $pattern) {
      if (preg_match($pattern, $query)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Validates if an introspection query is allowed.
   *
   * @param string $query
   *   The GraphQL query.
   *
   * @return array
   *   Array with 'allowed' boolean and optional 'error' message.
   */
  public function validate($query) {
    if (!$this->containsIntrospection($query)) {
      return ['allowed' => TRUE];
    }

    if ($this->isAllowed()) {
      return ['allowed' => TRUE];
    }

    return [
      'allowed' => FALSE,
      'error' => 'Introspection queries are not allowed',
    ];
  }

  /**
   * Filters schema based on user permissions.
   *
   * This would be used to return different schema parts to different users.
   *
   * @param array $schema
   *   The full GraphQL schema.
   *
   * @return array
   *   Filtered schema.
   */
  public function filterSchema(array $schema) {
    $config = $this->configFactory->get('graphql_shield.settings');

    // If no filtering configured, return full schema.
    if (!$config->get('introspection.filter_by_role')) {
      return $schema;
    }

    // Get user roles.
    $user_roles = $this->currentUser->getRoles();

    // Filter schema based on role configuration.
    $role_schemas = $config->get('introspection.role_schemas') ?: [];

    // This is a simplified version - in production you'd implement
    // more sophisticated schema filtering.
    foreach ($user_roles as $role) {
      if (isset($role_schemas[$role])) {
        // Apply role-specific schema modifications.
        $schema = $this->applySchemaFilter($schema, $role_schemas[$role]);
      }
    }

    return $schema;
  }

  /**
   * Applies schema filter for a specific role.
   *
   * @param array $schema
   *   The schema.
   * @param array $filter
   *   Filter configuration.
   *
   * @return array
   *   Filtered schema.
   */
  protected function applySchemaFilter(array $schema, array $filter) {
    // Implementation would filter out types, fields, etc.
    // based on role permissions.
    return $schema;
  }

}
