<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Symfony\Component\HttpFoundation\RequestStack;
use Psr\Log\LoggerInterface;

/**
 * Service for IP-based restrictions.
 */
class IpRestrictor {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an IpRestrictor object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    RequestStack $request_stack,
    LoggerInterface $logger,
  ) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->requestStack = $request_stack;
    $this->logger = $logger;
  }

  /**
   * Checks if an IP is allowed.
   *
   * @param string|null $ip
   *   IP address (uses current request IP if NULL).
   *
   * @return array
   *   Array with 'allowed' boolean and optional 'reason'.
   */
  public function isAllowed($ip = NULL) {
    if ($ip === NULL) {
      $request = $this->requestStack->getCurrentRequest();
      $ip = $request ? $request->getClientIp() : '0.0.0.0';
    }

    $config = $this->configFactory->get('graphql_shield.settings');

    // Check if IP restrictions are enabled.
    if (!$config->get('ip_restrictions.enabled')) {
      return ['allowed' => TRUE];
    }

    // Check blocklist first.
    if ($this->isBlocked($ip)) {
      $this->logger->warning('Blocked IP attempted access: @ip', ['@ip' => $ip]);
      return ['allowed' => FALSE, 'reason' => 'IP address is blocked'];
    }

    // If allowlist is enabled, check if IP is in allowlist.
    if ($config->get('ip_restrictions.use_allowlist')) {
      if (!$this->isAllowlisted($ip)) {
        $this->logger->warning('Non-allowlisted IP attempted access: @ip', ['@ip' => $ip]);
        return ['allowed' => FALSE, 'reason' => 'IP address not in allowlist'];
      }
    }

    return ['allowed' => TRUE];
  }

  /**
   * Checks if IP is blocked.
   *
   * @param string $ip
   *   IP address.
   *
   * @return bool
   *   TRUE if blocked.
   */
  public function isBlocked($ip) {
    // Clean up expired blocks first.
    $this->database->delete('graphql_shield_ip_rules')
      ->condition('rule_type', 'block')
      ->condition('expires', time(), '<')
      ->isNotNull('expires')
      ->execute();

    $count = $this->database->select('graphql_shield_ip_rules', 'r')
      ->condition('ip_address', $ip)
      ->condition('rule_type', 'block')
      ->countQuery()
      ->execute()
      ->fetchField();

    return $count > 0;
  }

  /**
   * Checks if IP is in allowlist.
   *
   * @param string $ip
   *   IP address.
   *
   * @return bool
   *   TRUE if allowlisted.
   */
  public function isAllowlisted($ip) {
    $count = $this->database->select('graphql_shield_ip_rules', 'r')
      ->condition('ip_address', $ip)
      ->condition('rule_type', 'allow')
      ->countQuery()
      ->execute()
      ->fetchField();

    return $count > 0;
  }

  /**
   * Blocks an IP address.
   *
   * @param string $ip
   *   IP address.
   * @param string|null $reason
   *   Reason for blocking.
   * @param int|null $duration
   *   Duration in seconds (NULL for permanent).
   * @param bool $auto_created
   *   Whether this was auto-created.
   */
  public function blockIp($ip, $reason = NULL, $duration = NULL, $auto_created = FALSE) {
    $this->database->insert('graphql_shield_ip_rules')
      ->fields([
        'ip_address' => $ip,
        'rule_type' => 'block',
        'reason' => $reason,
        'created' => time(),
        'expires' => $duration ? time() + $duration : NULL,
        'auto_created' => $auto_created ? 1 : 0,
      ])
      ->execute();

    $this->logger->notice('IP blocked: @ip (reason: @reason)', [
      '@ip' => $ip,
      '@reason' => $reason ?? 'Manual block',
    ]);
  }

  /**
   * Adds IP to allowlist.
   *
   * @param string $ip
   *   IP address.
   * @param string|null $reason
   *   Reason for allowing.
   */
  public function allowIp($ip, $reason = NULL) {
    $this->database->insert('graphql_shield_ip_rules')
      ->fields([
        'ip_address' => $ip,
        'rule_type' => 'allow',
        'reason' => $reason,
        'created' => time(),
        'expires' => NULL,
        'auto_created' => 0,
      ])
      ->execute();

    $this->logger->notice('IP added to allowlist: @ip', ['@ip' => $ip]);
  }

  /**
   * Removes IP rule.
   *
   * @param int $id
   *   Rule ID.
   */
  public function removeRule($id) {
    $this->database->delete('graphql_shield_ip_rules')
      ->condition('id', $id)
      ->execute();
  }

  /**
   * Unblocks an IP address.
   *
   * @param string $ip
   *   IP address.
   */
  public function unblockIp($ip) {
    $this->database->delete('graphql_shield_ip_rules')
      ->condition('ip_address', $ip)
      ->condition('rule_type', 'block')
      ->execute();

    $this->logger->notice('IP unblocked: @ip', ['@ip' => $ip]);
  }

  /**
   * Lists IP rules.
   *
   * @param string|null $type
   *   Rule type filter ('allow' or 'block').
   *
   * @return array
   *   Array of rules.
   */
  public function listRules($type = NULL) {
    $query = $this->database->select('graphql_shield_ip_rules', 'r')
      ->fields('r')
      ->orderBy('created', 'DESC');

    if ($type) {
      $query->condition('rule_type', $type);
    }

    return $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Auto-blocks an IP after repeated violations.
   *
   * @param string $ip
   *   IP address.
   * @param string $reason
   *   Violation reason.
   */
  public function autoBlock($ip, $reason) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('ip_restrictions.auto_block_enabled')) {
      return;
    }

    $duration = $config->get('ip_restrictions.auto_block_duration') ?: 3600;

    $this->blockIp($ip, "Auto-blocked: $reason", $duration, TRUE);
  }

}
