<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Service for managing persisted GraphQL queries.
 */
class PersistedQueryManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * Constructs a PersistedQueryManager object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    CacheBackendInterface $cache,
  ) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->cache = $cache;
  }

  /**
   * Checks if persisted queries are enabled.
   *
   * @return bool
   *   TRUE if enabled.
   */
  public function isEnabled() {
    $config = $this->configFactory->get('graphql_shield.settings');
    return (bool) $config->get('persisted_queries.enabled');
  }

  /**
   * Checks if in development mode (allows non-persisted queries).
   *
   * @return bool
   *   TRUE if in development mode.
   */
  public function isDevelopmentMode() {
    $config = $this->configFactory->get('graphql_shield.settings');
    return (bool) $config->get('persisted_queries.development_mode');
  }

  /**
   * Gets a persisted query by hash or ID.
   *
   * @param string $identifier
   *   Query hash or ID.
   *
   * @return array|null
   *   Query data or NULL if not found.
   */
  public function getQuery($identifier) {
    // Check cache first.
    $cache_key = 'graphql_shield:persisted_query:' . $identifier;
    $cached = $this->cache->get($cache_key);

    if ($cached) {
      return $cached->data;
    }

    // Query database.
    $query = $this->database->select('graphql_shield_persisted_queries', 'q')
      ->fields('q')
      ->condition('enabled', 1)
      ->where('query_hash = :id OR query_id = :id', [':id' => $identifier])
      ->execute()
      ->fetchAssoc();

    if ($query) {
      // Cache for 1 hour.
      $this->cache->set($cache_key, $query, time() + 3600);

      // Increment usage count.
      $this->database->update('graphql_shield_persisted_queries')
        ->expression('usage_count', 'usage_count + 1')
        ->condition('id', $query['id'])
        ->execute();
    }

    return $query ?: NULL;
  }

  /**
   * Validates if a query is allowed.
   *
   * @param string $query
   *   The GraphQL query string.
   * @param string|null $query_id
   *   Optional query ID from client.
   *
   * @return array
   *   Array with 'allowed' boolean and 'query' if found.
   */
  public function validateQuery($query, $query_id = NULL) {
    if (!$this->isEnabled()) {
      return ['allowed' => TRUE, 'query' => $query];
    }

    // Development mode allows all queries.
    if ($this->isDevelopmentMode()) {
      return ['allowed' => TRUE, 'query' => $query];
    }

    // If query_id provided, look it up.
    if ($query_id) {
      $persisted = $this->getQuery($query_id);

      if ($persisted) {
        return ['allowed' => TRUE, 'query' => $persisted['query']];
      }
    }

    // Check if query hash matches any persisted query.
    $query_hash = hash('sha256', trim($query));
    $persisted = $this->getQuery($query_hash);

    if ($persisted) {
      return ['allowed' => TRUE, 'query' => $persisted['query']];
    }

    // Query not found in persisted list.
    return ['allowed' => FALSE, 'error' => 'Query not allowed. Only persisted queries are permitted.'];
  }

  /**
   * Adds a new persisted query.
   *
   * @param string $query_id
   *   Human-readable query ID.
   * @param string $query
   *   The GraphQL query.
   * @param string|null $description
   *   Optional description.
   *
   * @return int
   *   The query ID.
   */
  public function addQuery($query_id, $query, $description = NULL) {
    $query_hash = hash('sha256', trim($query));

    // Check if already exists.
    $existing = $this->database->select('graphql_shield_persisted_queries', 'q')
      ->fields('q', ['id'])
      ->condition('query_hash', $query_hash)
      ->execute()
      ->fetchField();

    if ($existing) {
      return $existing;
    }

    return $this->database->insert('graphql_shield_persisted_queries')
      ->fields([
        'query_hash' => $query_hash,
        'query_id' => $query_id,
        'query' => $query,
        'description' => $description,
        'enabled' => 1,
        'created' => time(),
        'usage_count' => 0,
      ])
      ->execute();
  }

  /**
   * Removes a persisted query.
   *
   * @param int $id
   *   Query database ID.
   */
  public function removeQuery($id) {
    $this->database->delete('graphql_shield_persisted_queries')
      ->condition('id', $id)
      ->execute();

    // Clear cache.
    $this->cache->deleteAll();
  }

  /**
   * Lists all persisted queries.
   *
   * @return array
   *   Array of queries.
   */
  public function listQueries() {
    return $this->database->select('graphql_shield_persisted_queries', 'q')
      ->fields('q')
      ->orderBy('created', 'DESC')
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Enables/disables a persisted query.
   *
   * @param int $id
   *   Query ID.
   * @param bool $enabled
   *   Enable or disable.
   */
  public function setEnabled($id, $enabled) {
    $this->database->update('graphql_shield_persisted_queries')
      ->fields(['enabled' => $enabled ? 1 : 0])
      ->condition('id', $id)
      ->execute();

    // Clear cache.
    $this->cache->deleteAll();
  }

}
