<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for advanced query analysis and blocking.
 */
class QueryAnalyzer {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a QueryAnalyzer object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerInterface $logger) {
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * Analyzes a query for suspicious patterns.
   *
   * @param string $query
   *   The GraphQL query.
   *
   * @return array
   *   Analysis result with 'safe' boolean and optional 'issues' array.
   */
  public function analyze($query) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('query_analysis.enabled')) {
      return ['safe' => TRUE];
    }

    $issues = [];

    // Check for regex-based blocking patterns.
    if ($blocked = $this->checkBlockedPatterns($query)) {
      $issues[] = "Query matches blocked pattern: $blocked";
    }

    // Check for excessive aliases.
    if ($this->checkExcessiveAliases($query)) {
      $issues[] = 'Query contains excessive aliases';
    }

    // Check for directive abuse.
    if ($this->checkDirectiveAbuse($query)) {
      $issues[] = 'Query contains excessive directives';
    }

    // Check for fragment depth.
    if ($this->checkFragmentDepth($query)) {
      $issues[] = 'Query fragments are too deeply nested';
    }

    // Check for circular fragments.
    if ($this->checkCircularFragments($query)) {
      $issues[] = 'Query contains circular fragment references';
    }

    if (!empty($issues)) {
      $this->logger->warning('Suspicious query detected: @issues', [
        '@issues' => implode(', ', $issues),
      ]);

      return ['safe' => FALSE, 'issues' => $issues];
    }

    return ['safe' => TRUE];
  }

  /**
   * Checks for blocked regex patterns.
   *
   * @param string $query
   *   The query.
   *
   * @return string|false
   *   Matched pattern or FALSE.
   */
  protected function checkBlockedPatterns($query) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $patterns = $config->get('query_analysis.blocked_patterns') ?: [];

    foreach ($patterns as $pattern) {
      if (@preg_match($pattern, $query)) {
        return $pattern;
      }
    }

    return FALSE;
  }

  /**
   * Checks for excessive aliases.
   *
   * @param string $query
   *   The query.
   *
   * @return bool
   *   TRUE if excessive.
   */
  protected function checkExcessiveAliases($query) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $max = $config->get('query_analysis.max_aliases') ?: 50;

    $count = preg_match_all('/\b\w+\s*:\s*\w+/', $query);

    return $count > $max;
  }

  /**
   * Checks for directive abuse.
   *
   * @param string $query
   *   The query.
   *
   * @return bool
   *   TRUE if abuse detected.
   */
  protected function checkDirectiveAbuse($query) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $max = $config->get('query_analysis.max_directives') ?: 20;

    $count = preg_match_all('/@\w+/', $query);

    return $count > $max;
  }

  /**
   * Checks fragment nesting depth.
   *
   * @param string $query
   *   The query.
   *
   * @return bool
   *   TRUE if too deep.
   */
  protected function checkFragmentDepth($query) {
    $config = $this->configFactory->get('graphql_shield.settings');
    $max_depth = $config->get('query_analysis.max_fragment_depth') ?: 5;

    // Extract all fragments.
    preg_match_all('/fragment\s+(\w+).*?\{(.*?)\}/s', $query, $matches);

    if (empty($matches[1])) {
      return FALSE;
    }

    $fragments = array_combine($matches[1], $matches[2]);

    // Check depth of each fragment.
    foreach ($fragments as $name => $content) {
      $depth = $this->calculateFragmentDepth($name, $content, $fragments);

      if ($depth > $max_depth) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Calculates fragment nesting depth.
   *
   * @param string $fragment_name
   *   Fragment name.
   * @param string $content
   *   Fragment content.
   * @param array $all_fragments
   *   All fragments.
   * @param int $current_depth
   *   Current depth.
   *
   * @return int
   *   Depth.
   */
  protected function calculateFragmentDepth($fragment_name, $content, array $all_fragments, $current_depth = 0) {
    $max_depth = $current_depth;

    // Find fragment spreads in content.
    preg_match_all('/\.\.\.\s*(\w+)/', $content, $matches);

    foreach ($matches[1] as $spread_name) {
      if (isset($all_fragments[$spread_name])) {
        $depth = $this->calculateFragmentDepth(
          $spread_name,
          $all_fragments[$spread_name],
          $all_fragments,
          $current_depth + 1
        );
        $max_depth = max($max_depth, $depth);
      }
    }

    return $max_depth;
  }

  /**
   * Checks for circular fragment references.
   *
   * @param string $query
   *   The query.
   *
   * @return bool
   *   TRUE if circular references found.
   */
  protected function checkCircularFragments($query) {
    preg_match_all('/fragment\s+(\w+).*?\{(.*?)\}/s', $query, $matches);

    if (empty($matches[1])) {
      return FALSE;
    }

    $fragments = array_combine($matches[1], $matches[2]);

    foreach ($fragments as $name => $content) {
      if ($this->hasCircularReference($name, $content, $fragments)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Checks if a fragment has circular references.
   *
   * @param string $fragment_name
   *   Fragment name.
   * @param string $content
   *   Fragment content.
   * @param array $all_fragments
   *   All fragments.
   * @param array $visited
   *   Visited fragments.
   *
   * @return bool
   *   TRUE if circular.
   */
  protected function hasCircularReference($fragment_name, $content, array $all_fragments, array $visited = []) {
    if (in_array($fragment_name, $visited)) {
      return TRUE;
    }

    $visited[] = $fragment_name;

    preg_match_all('/\.\.\.\s*(\w+)/', $content, $matches);

    foreach ($matches[1] as $spread_name) {
      if (isset($all_fragments[$spread_name])) {
        if ($this->hasCircularReference($spread_name, $all_fragments[$spread_name], $all_fragments, $visited)) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

}
