<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for analyzing and limiting GraphQL query complexity.
 */
class QueryComplexityAnalyzer {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a QueryComplexityAnalyzer object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerInterface $logger) {
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * Analyzes query complexity.
   *
   * @param string $query
   *   The GraphQL query string.
   * @param array $variables
   *   Query variables.
   *
   * @return array
   *   Array containing 'depth', 'complexity', and 'allowed' keys.
   */
  public function analyze($query, array $variables = []) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('query_complexity.enabled')) {
      return ['allowed' => TRUE, 'depth' => 0, 'complexity' => 0];
    }

    $depth = $this->calculateDepth($query);
    $complexity = $this->calculateComplexity($query, $variables);

    $max_depth = $config->get('query_complexity.max_depth') ?: 10;
    $max_complexity = $config->get('query_complexity.max_complexity') ?: 1000;

    $allowed = $depth <= $max_depth && $complexity <= $max_complexity;

    if (!$allowed) {
      $this->logger->warning('Query complexity exceeded: depth=@depth (max @max_depth), complexity=@complexity (max @max_complexity)', [
        '@depth' => $depth,
        '@max_depth' => $max_depth,
        '@complexity' => $complexity,
        '@max_complexity' => $max_complexity,
      ]);
    }

    return [
      'allowed' => $allowed,
      'depth' => $depth,
      'complexity' => $complexity,
      'max_depth' => $max_depth,
      'max_complexity' => $max_complexity,
    ];
  }

  /**
   * Calculates query depth.
   *
   * @param string $query
   *   The GraphQL query.
   *
   * @return int
   *   The maximum depth.
   */
  protected function calculateDepth($query) {
    // Remove comments and strings to avoid false positives.
    $cleaned = preg_replace('/#.*$/m', '', $query);
    $cleaned = preg_replace('/"[^"]*"/', '""', $cleaned);

    $max_depth = 0;
    $current_depth = 0;

    for ($i = 0; $i < strlen($cleaned); $i++) {
      if ($cleaned[$i] === '{') {
        $current_depth++;
        $max_depth = max($max_depth, $current_depth);
      }
      elseif ($cleaned[$i] === '}') {
        $current_depth--;
      }
    }

    return $max_depth;
  }

  /**
   * Calculates query complexity score.
   *
   * @param string $query
   *   The GraphQL query.
   * @param array $variables
   *   Query variables.
   *
   * @return int
   *   The complexity score.
   */
  protected function calculateComplexity($query, array $variables) {
    $complexity = 0;

    // Base complexity for the query.
    $complexity += 1;

    // Count fields (each field adds 1).
    $field_count = preg_match_all('/\b\w+\s*[:{]/', $query, $matches);
    $complexity += $field_count;

    // Aliases add complexity (2 points each).
    $alias_count = preg_match_all('/\b\w+\s*:\s*\w+/', $query, $matches);
    $complexity += $alias_count * 2;

    // Fragments add complexity (5 points each).
    $fragment_count = preg_match_all('/\.\.\.\s*\w+/', $query, $matches);
    $complexity += $fragment_count * 5;

    // Directives add complexity (3 points each).
    $directive_count = preg_match_all('/@\w+/', $query, $matches);
    $complexity += $directive_count * 3;

    // Lists with pagination multiply complexity.
    if (preg_match_all('/\b(first|last|limit)\s*:\s*(\d+|\$\w+)/', $query, $matches)) {
      foreach ($matches[2] as $limit) {
        if (strpos($limit, '$') === 0) {
          // Variable reference - check in variables.
          $var_name = substr($limit, 1);
          $limit_value = $variables[$var_name] ?? 10;
        }
        else {
          $limit_value = (int) $limit;
        }
        $complexity += ceil($limit_value / 10);
      }
    }

    // Nested queries multiply complexity.
    $depth = $this->calculateDepth($query);
    $complexity *= (1 + ($depth * 0.5));

    return (int) $complexity;
  }

  /**
   * Checks if a query is allowed based on complexity.
   *
   * @param string $query
   *   The GraphQL query.
   * @param array $variables
   *   Query variables.
   *
   * @return bool
   *   TRUE if allowed, FALSE otherwise.
   */
  public function isAllowed($query, array $variables = []) {
    $analysis = $this->analyze($query, $variables);
    return $analysis['allowed'];
  }

}
