<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Symfony\Component\HttpFoundation\RequestStack;
use Psr\Log\LoggerInterface;

/**
 * Service for rate limiting GraphQL requests.
 */
class RateLimiter {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a RateLimiter object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Connection $database,
    RequestStack $request_stack,
    LoggerInterface $logger,
  ) {
    $this->configFactory = $config_factory;
    $this->database = $database;
    $this->requestStack = $request_stack;
    $this->logger = $logger;
  }

  /**
   * Checks if a request is allowed based on rate limits.
   *
   * @param int $uid
   *   User ID.
   * @param string $ip
   *   IP address.
   *
   * @return array
   *   Array with 'allowed' boolean and 'retry_after' seconds.
   */
  public function checkLimit($uid, $ip) {
    $config = $this->configFactory->get('graphql_shield.settings');

    if (!$config->get('rate_limiting.enabled')) {
      return ['allowed' => TRUE, 'retry_after' => 0];
    }

    $time_window = $config->get('rate_limiting.time_window') ?: 60;
    $current_time = time();

    // Check user rate limit.
    if ($uid > 0) {
      $user_limit = $config->get('rate_limiting.per_user') ?: 100;
      $user_allowed = $this->checkIdentifier('user', (string) $uid, $user_limit, $time_window, $current_time);

      if (!$user_allowed['allowed']) {
        $this->logger->warning('Rate limit exceeded for user @uid', ['@uid' => $uid]);
        return $user_allowed;
      }
    }

    // Check IP rate limit.
    $ip_limit = $config->get('rate_limiting.per_ip') ?: 60;
    $ip_allowed = $this->checkIdentifier('ip', $ip, $ip_limit, $time_window, $current_time);

    if (!$ip_allowed['allowed']) {
      $this->logger->warning('Rate limit exceeded for IP @ip', ['@ip' => $ip]);
      return $ip_allowed;
    }

    return ['allowed' => TRUE, 'retry_after' => 0];
  }

  /**
   * Checks rate limit for a specific identifier.
   *
   * @param string $type
   *   Identifier type ('user' or 'ip').
   * @param string $identifier
   *   The identifier value.
   * @param int $limit
   *   Request limit.
   * @param int $time_window
   *   Time window in seconds.
   * @param int $current_time
   *   Current timestamp.
   *
   * @return array
   *   Array with 'allowed' and 'retry_after'.
   */
  protected function checkIdentifier($type, $identifier, $limit, $time_window, $current_time) {
    $window_start = $current_time - $time_window;

    // Clean up old entries.
    $this->database->delete('graphql_shield_rate_limits')
      ->condition('window_end', $current_time, '<')
      ->execute();

    // Get or create rate limit record.
    $record = $this->database->select('graphql_shield_rate_limits', 'r')
      ->fields('r')
      ->condition('identifier', $identifier)
      ->condition('identifier_type', $type)
      ->condition('window_end', $current_time, '>=')
      ->execute()
      ->fetchAssoc();

    if (!$record) {
      // Create new record.
      $this->database->insert('graphql_shield_rate_limits')
        ->fields([
          'identifier' => $identifier,
          'identifier_type' => $type,
          'request_count' => 1,
          'window_start' => $current_time,
          'window_end' => $current_time + $time_window,
          'last_request' => $current_time,
        ])
        ->execute();

      return ['allowed' => TRUE, 'retry_after' => 0, 'remaining' => $limit - 1];
    }

    // Check if limit exceeded.
    if ($record['request_count'] >= $limit) {
      $retry_after = $record['window_end'] - $current_time;
      return [
        'allowed' => FALSE,
        'retry_after' => $retry_after,
        'remaining' => 0,
      ];
    }

    // Increment counter.
    $this->database->update('graphql_shield_rate_limits')
      ->fields([
        'request_count' => $record['request_count'] + 1,
        'last_request' => $current_time,
      ])
      ->condition('id', $record['id'])
      ->execute();

    return [
      'allowed' => TRUE,
      'retry_after' => 0,
      'remaining' => $limit - ($record['request_count'] + 1),
    ];
  }

  /**
   * Records a request for rate limiting.
   *
   * @param int $uid
   *   User ID.
   * @param string $ip
   *   IP address.
   */
  public function recordRequest($uid, $ip) {
    $this->checkLimit($uid, $ip);
  }

  /**
   * Resets rate limit for an identifier.
   *
   * @param string $type
   *   Type ('user' or 'ip').
   * @param string $identifier
   *   The identifier.
   */
  public function resetLimit($type, $identifier) {
    $this->database->delete('graphql_shield_rate_limits')
      ->condition('identifier', $identifier)
      ->condition('identifier_type', $type)
      ->execute();
  }

  /**
   * Gets current rate limit status.
   *
   * @param string $type
   *   Type ('user' or 'ip').
   * @param string $identifier
   *   The identifier.
   *
   * @return array
   *   Status information.
   */
  public function getStatus($type, $identifier) {
    $record = $this->database->select('graphql_shield_rate_limits', 'r')
      ->fields('r')
      ->condition('identifier', $identifier)
      ->condition('identifier_type', $type)
      ->condition('window_end', time(), '>=')
      ->execute()
      ->fetchAssoc();

    if (!$record) {
      return [
        'request_count' => 0,
        'window_remaining' => 0,
      ];
    }

    return [
      'request_count' => $record['request_count'],
      'window_remaining' => $record['window_end'] - time(),
    ];
  }

}
