<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for validating GraphQL request size and parameters.
 */
class RequestValidator {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a RequestValidator object.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerInterface $logger) {
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * Validates a GraphQL request.
   *
   * @param string $query
   *   The query string.
   * @param array $variables
   *   Query variables.
   * @param array $operations
   *   Batch operations.
   *
   * @return array
   *   Array with 'valid' boolean and optional 'error' message.
   */
  public function validate($query, array $variables = [], array $operations = []) {
    $config = $this->configFactory->get('graphql_shield.settings');

    // Check query size.
    $max_query_size = $config->get('request_limits.max_query_size') ?: 10000;
    if (strlen($query) > $max_query_size) {
      $this->logger->warning('Query size exceeded: @size bytes (max @max)', [
        '@size' => strlen($query),
        '@max' => $max_query_size,
      ]);

      return [
        'valid' => FALSE,
        'error' => sprintf('Query size exceeds maximum allowed (%d bytes)', $max_query_size),
      ];
    }

    // Check number of variables.
    $max_variables = $config->get('request_limits.max_variables') ?: 50;
    if (count($variables) > $max_variables) {
      $this->logger->warning('Too many variables: @count (max @max)', [
        '@count' => count($variables),
        '@max' => $max_variables,
      ]);

      return [
        'valid' => FALSE,
        'error' => sprintf('Too many query variables (max %d)', $max_variables),
      ];
    }

    // Check batch query count.
    $max_batch = $config->get('request_limits.max_batch_queries') ?: 10;
    if (count($operations) > $max_batch) {
      $this->logger->warning('Too many batch operations: @count (max @max)', [
        '@count' => count($operations),
        '@max' => $max_batch,
      ]);

      return [
        'valid' => FALSE,
        'error' => sprintf('Too many batch operations (max %d)', $max_batch),
      ];
    }

    // Validate query syntax (basic check).
    if (!$this->validateQuerySyntax($query)) {
      return [
        'valid' => FALSE,
        'error' => 'Invalid query syntax',
      ];
    }

    return ['valid' => TRUE];
  }

  /**
   * Validates basic query syntax.
   *
   * @param string $query
   *   The query.
   *
   * @return bool
   *   TRUE if valid.
   */
  protected function validateQuerySyntax($query) {
    // Check for balanced braces.
    $open_braces = substr_count($query, '{');
    $close_braces = substr_count($query, '}');

    if ($open_braces !== $close_braces) {
      $this->logger->warning('Unbalanced braces in query');
      return FALSE;
    }

    // Check for balanced parentheses.
    $open_parens = substr_count($query, '(');
    $close_parens = substr_count($query, ')');

    if ($open_parens !== $close_parens) {
      $this->logger->warning('Unbalanced parentheses in query');
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Sets timeout for query execution.
   *
   * @return int
   *   Timeout in seconds.
   */
  public function getTimeout() {
    $config = $this->configFactory->get('graphql_shield.settings');
    return $config->get('request_limits.timeout') ?: 30;
  }

  /**
   * Gets memory limit for query execution.
   *
   * @return string
   *   Memory limit (e.g., '128M').
   */
  public function getMemoryLimit() {
    $config = $this->configFactory->get('graphql_shield.settings');
    return $config->get('request_limits.memory_limit') ?: '128M';
  }

}
