<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for security dashboard data.
 */
class SecurityDashboard {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a SecurityDashboard object.
   */
  public function __construct(Connection $database, ConfigFactoryInterface $config_factory) {
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * Gets dashboard statistics.
   *
   * @param int $time_period
   *   Time period in seconds (default 24 hours).
   *
   * @return array
   *   Dashboard data.
   */
  public function getDashboardData($time_period = 86400) {
    $start_time = time() - $time_period;

    return [
      'overview' => $this->getOverviewStats($start_time),
      'top_users' => $this->getTopUsers($start_time),
      'top_ips' => $this->getTopIps($start_time),
      'recent_blocks' => $this->getRecentBlocks(10),
      'slow_queries' => $this->getSlowQueries($start_time, 10),
      'error_distribution' => $this->getErrorDistribution($start_time),
      'hourly_stats' => $this->getHourlyStats($time_period),
    ];
  }

  /**
   * Gets overview statistics.
   *
   * @param int $start_time
   *   Start timestamp.
   *
   * @return array
   *   Overview stats.
   */
  protected function getOverviewStats($start_time) {
    return [
      'total_requests' => $this->countRequests($start_time),
      'successful_queries' => $this->countSuccessfulQueries($start_time),
      'blocked_requests' => $this->countBlockedRequests($start_time),
      'errors' => $this->countErrors($start_time),
      'unique_users' => $this->countUniqueUsers($start_time),
      'unique_ips' => $this->countUniqueIps($start_time),
      'avg_response_time' => $this->getAvgResponseTime($start_time),
      'api_keys_active' => $this->countActiveApiKeys(),
      'persisted_queries' => $this->countPersistedQueries(),
      'ip_blocks' => $this->countIpBlocks(),
    ];
  }

  /**
   * Counts total requests.
   */
  protected function countRequests($start_time) {
    return $this->database->select('graphql_shield_logs', 'l')
      ->condition('timestamp', $start_time, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Counts successful queries.
   */
  protected function countSuccessfulQueries($start_time) {
    return $this->database->select('graphql_shield_logs', 'l')
      ->condition('timestamp', $start_time, '>=')
      ->condition('event_type', 'query')
      ->condition('severity', 'info')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Counts blocked requests.
   */
  protected function countBlockedRequests($start_time) {
    return $this->database->select('graphql_shield_logs', 'l')
      ->condition('timestamp', $start_time, '>=')
      ->condition('event_type', 'blocked')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Counts errors.
   */
  protected function countErrors($start_time) {
    return $this->database->select('graphql_shield_logs', 'l')
      ->condition('timestamp', $start_time, '>=')
      ->condition('severity', ['error', 'critical'], 'IN')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Counts unique users.
   */
  protected function countUniqueUsers($start_time) {
    return $this->database->select('graphql_shield_logs', 'l')
      ->distinct()
      ->fields('l', ['uid'])
      ->condition('timestamp', $start_time, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Counts unique IPs.
   */
  protected function countUniqueIps($start_time) {
    return $this->database->select('graphql_shield_logs', 'l')
      ->distinct()
      ->fields('l', ['ip_address'])
      ->condition('timestamp', $start_time, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Gets average response time.
   */
  protected function getAvgResponseTime($start_time) {
    $query = $this->database->select('graphql_shield_logs', 'l');
    $query->addExpression('AVG(execution_time)', 'avg_time');
    $query->condition('timestamp', $start_time, '>=');
    $query->isNotNull('execution_time');

    $result = $query->execute()->fetchField();
    return $result ? round($result, 3) : 0;
  }

  /**
   * Counts active API keys.
   */
  protected function countActiveApiKeys() {
    return $this->database->select('graphql_shield_api_keys', 'k')
      ->condition('enabled', 1)
      ->condition('expires', time(), '>')
      ->isNotNull('expires')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Counts persisted queries.
   */
  protected function countPersistedQueries() {
    return $this->database->select('graphql_shield_persisted_queries', 'q')
      ->condition('enabled', 1)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Counts IP blocks.
   */
  protected function countIpBlocks() {
    return $this->database->select('graphql_shield_ip_rules', 'r')
      ->condition('rule_type', 'block')
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Gets top users by request count.
   */
  protected function getTopUsers($start_time, $limit = 10) {
    $query = $this->database->select('graphql_shield_logs', 'l');
    $query->addField('l', 'uid');
    $query->addExpression('COUNT(*)', 'request_count');
    $query->condition('timestamp', $start_time, '>=');
    $query->groupBy('uid');
    $query->orderBy('request_count', 'DESC');
    $query->range(0, $limit);

    return $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Gets top IPs by request count.
   */
  protected function getTopIps($start_time, $limit = 10) {
    $query = $this->database->select('graphql_shield_logs', 'l');
    $query->addField('l', 'ip_address');
    $query->addExpression('COUNT(*)', 'request_count');
    $query->condition('timestamp', $start_time, '>=');
    $query->groupBy('ip_address');
    $query->orderBy('request_count', 'DESC');
    $query->range(0, $limit);

    return $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Gets recent blocked requests.
   */
  protected function getRecentBlocks($limit = 10) {
    return $this->database->select('graphql_shield_logs', 'l')
      ->fields('l')
      ->condition('event_type', 'blocked')
      ->orderBy('timestamp', 'DESC')
      ->range(0, $limit)
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Gets slow queries.
   */
  protected function getSlowQueries($start_time, $limit = 10) {
    return $this->database->select('graphql_shield_logs', 'l')
      ->fields('l')
      ->condition('timestamp', $start_time, '>=')
      ->condition('execution_time', 1, '>')
      ->isNotNull('execution_time')
      ->orderBy('execution_time', 'DESC')
      ->range(0, $limit)
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Gets error distribution by type.
   */
  protected function getErrorDistribution($start_time) {
    $query = $this->database->select('graphql_shield_logs', 'l');
    $query->addField('l', 'event_type');
    $query->addExpression('COUNT(*)', 'count');
    $query->condition('timestamp', $start_time, '>=');
    $query->condition('severity', ['error', 'critical'], 'IN');
    $query->groupBy('event_type');

    return $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Gets hourly statistics.
   */
  protected function getHourlyStats($time_period) {
    $start_time = time() - $time_period;

    // Get all logs within the time period.
    $logs = $this->database->select('graphql_shield_logs', 'l')
      ->fields('l', ['timestamp', 'event_type', 'severity'])
      ->condition('timestamp', $start_time, '>=')
      ->execute()
      ->fetchAll(\PDO::FETCH_ASSOC);

    // Group by hour.
    $hourly_data = [];
    foreach ($logs as $log) {
      $hour = date('Y-m-d H:00:00', $log['timestamp']);

      if (!isset($hourly_data[$hour])) {
        $hourly_data[$hour] = [
          'hour' => $hour,
          'total' => 0,
          'blocked' => 0,
          'errors' => 0,
        ];
      }

      $hourly_data[$hour]['total']++;

      if ($log['event_type'] === 'blocked') {
        $hourly_data[$hour]['blocked']++;
      }

      if (in_array($log['severity'], ['error', 'critical'])) {
        $hourly_data[$hour]['errors']++;
      }
    }

    // Sort by hour.
    ksort($hourly_data);

    return array_values($hourly_data);
  }

}
