<?php

namespace Drupal\graphql_shield\Service;

use Drupal\Core\Database\Connection;
use Psr\Log\LoggerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for security logging and auditing.
 */
class SecurityLogger {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a SecurityLogger object.
   */
  public function __construct(
    Connection $database,
    LoggerInterface $logger,
    AccountProxyInterface $current_user,
    RequestStack $request_stack,
  ) {
    $this->database = $database;
    $this->logger = $logger;
    $this->currentUser = $current_user;
    $this->requestStack = $request_stack;
  }

  /**
   * Logs a security event.
   *
   * @param string $event_type
   *   Event type (query, error, blocked, etc).
   * @param string $severity
   *   Severity (info, warning, error, critical).
   * @param array $data
   *   Additional data to log.
   */
  public function logEvent($event_type, $severity, array $data = []) {
    $request = $this->requestStack->getCurrentRequest();
    $ip_address = $request ? $request->getClientIp() : '0.0.0.0';

    $this->database->insert('graphql_shield_logs')
      ->fields([
        'timestamp' => time(),
        'uid' => $this->currentUser->id(),
        'ip_address' => $ip_address,
        'event_type' => $event_type,
        'severity' => $severity,
        'query' => $data['query'] ?? NULL,
        'variables' => isset($data['variables']) ? json_encode($data['variables']) : NULL,
        'message' => $data['message'] ?? NULL,
        'execution_time' => $data['execution_time'] ?? NULL,
        'complexity_score' => $data['complexity_score'] ?? NULL,
        'data' => isset($data['extra']) ? serialize($data['extra']) : NULL,
      ])
      ->execute();

    // Also log to Drupal's logger.
    if ($severity === 'error' || $severity === 'critical') {
      $this->logger->error('@type: @message', [
        '@type' => $event_type,
        '@message' => $data['message'] ?? 'GraphQL security event',
      ]);
    }
  }

  /**
   * Logs a GraphQL query.
   *
   * @param string $query
   *   The query.
   * @param array $variables
   *   Query variables.
   * @param float $execution_time
   *   Execution time.
   * @param int $complexity_score
   *   Complexity score.
   */
  public function logQuery($query, array $variables, $execution_time, $complexity_score) {
    $this->logEvent('query', 'info', [
      'query' => $query,
      'variables' => $variables,
      'execution_time' => $execution_time,
      'complexity_score' => $complexity_score,
      'message' => 'GraphQL query executed',
    ]);
  }

  /**
   * Logs a blocked request.
   *
   * @param string $reason
   *   Block reason.
   * @param array $context
   *   Additional context.
   */
  public function logBlocked($reason, array $context = []) {
    $this->logEvent('blocked', 'warning', [
      'message' => $reason,
      'query' => $context['query'] ?? NULL,
      'extra' => $context,
    ]);
  }

  /**
   * Logs an error.
   *
   * @param string $message
   *   Error message.
   * @param array $context
   *   Additional context.
   */
  public function logError($message, array $context = []) {
    $this->logEvent('error', 'error', [
      'message' => $message,
      'query' => $context['query'] ?? NULL,
      'extra' => $context,
    ]);
  }

  /**
   * Gets recent security logs.
   *
   * @param int $limit
   *   Number of logs to retrieve.
   * @param array $filters
   *   Filters (event_type, severity, uid, ip_address).
   *
   * @return array
   *   Array of log entries.
   */
  public function getLogs($limit = 100, array $filters = []) {
    $query = $this->database->select('graphql_shield_logs', 'l')
      ->fields('l')
      ->orderBy('timestamp', 'DESC')
      ->range(0, $limit);

    if (!empty($filters['event_type'])) {
      $query->condition('event_type', $filters['event_type']);
    }

    if (!empty($filters['severity'])) {
      $query->condition('severity', $filters['severity']);
    }

    if (!empty($filters['uid'])) {
      $query->condition('uid', $filters['uid']);
    }

    if (!empty($filters['ip_address'])) {
      $query->condition('ip_address', $filters['ip_address']);
    }

    return $query->execute()->fetchAll(\PDO::FETCH_ASSOC);
  }

  /**
   * Gets security statistics.
   *
   * @param int $time_period
   *   Time period in seconds (default 24 hours).
   *
   * @return array
   *   Statistics array.
   */
  public function getStatistics($time_period = 86400) {
    $start_time = time() - $time_period;

    $stats = [
      'total_queries' => 0,
      'blocked_requests' => 0,
      'errors' => 0,
      'unique_users' => 0,
      'unique_ips' => 0,
      'avg_execution_time' => 0,
      'slow_queries' => 0,
    ];

    // Total queries.
    $stats['total_queries'] = $this->database->select('graphql_shield_logs', 'l')
      ->condition('timestamp', $start_time, '>=')
      ->condition('event_type', 'query')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Blocked requests.
    $stats['blocked_requests'] = $this->database->select('graphql_shield_logs', 'l')
      ->condition('timestamp', $start_time, '>=')
      ->condition('event_type', 'blocked')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Errors.
    $stats['errors'] = $this->database->select('graphql_shield_logs', 'l')
      ->condition('timestamp', $start_time, '>=')
      ->condition('severity', ['error', 'critical'], 'IN')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Unique users.
    $stats['unique_users'] = $this->database->select('graphql_shield_logs', 'l')
      ->distinct()
      ->fields('l', ['uid'])
      ->condition('timestamp', $start_time, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Unique IPs.
    $stats['unique_ips'] = $this->database->select('graphql_shield_logs', 'l')
      ->distinct()
      ->fields('l', ['ip_address'])
      ->condition('timestamp', $start_time, '>=')
      ->countQuery()
      ->execute()
      ->fetchField();

    // Average execution time.
    $avg = $this->database->query(
      "SELECT AVG(execution_time) FROM {graphql_shield_logs} WHERE timestamp >= :start AND execution_time IS NOT NULL",
      [':start' => $start_time]
    )->fetchField();
    $stats['avg_execution_time'] = round($avg, 3);

    // Slow queries (> 5 seconds).
    $stats['slow_queries'] = $this->database->select('graphql_shield_logs', 'l')
      ->condition('timestamp', $start_time, '>=')
      ->condition('execution_time', 5, '>')
      ->countQuery()
      ->execute()
      ->fetchField();

    return $stats;
  }

  /**
   * Cleans up old logs.
   *
   * @param int $retention_period
   *   Retention period in seconds (default 30 days).
   */
  public function cleanupOldLogs($retention_period = 2592000) {
    $cutoff = time() - $retention_period;

    $this->database->delete('graphql_shield_logs')
      ->condition('timestamp', $cutoff, '<')
      ->execute();
  }

  /**
   * Clears all security logs.
   *
   * @return int
   *   Number of deleted log entries.
   */
  public function clearAllLogs() {
    return $this->database->delete('graphql_shield_logs')
      ->execute();
  }

}
