<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\taxonomy\TermStorageInterface;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns the options of a term select Webform element.
 *
 * @DataProducer(
 *   id = "webform_term_select_options",
 *   name = @Translation("Webform term select element options"),
 *   description = @Translation("Returns the available options for a term select element."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Any value"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "element" = @ContextDefinition(
 *       "any",
 *       label = @Translation("Webform element")
 *     ),
 *     "depth" = @ContextDefinition("integer", label = @Translation("Depth"))
 *   }
 * )
 */
class WebformTermSelectOptions extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The cache tags for the term list.
   *
   * @var string[]
   */
  protected array $cacheTags = ['taxonomy_term_list'];

  /**
   * Constructs a new WebformTermSelectOptions instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $pluginId
   *   The plugin ID.
   * @param mixed $pluginDefinition
   *   The plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\webform\Plugin\WebformElementManagerInterface $elementManager
   *   The Webform element plugin manager.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly WebformElementManagerInterface $elementManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.webform.element')
    );
  }

  /**
   * Resolves taxonomy term select options.
   *
   * @param array $element
   *   The Webform element for which to return the options.
   * @param int|null $depth
   *   Optional depth of the term tree to return. If omitted the depth that is
   *   configured in the Webform element will be used.
   *
   * @return array
   *   The terms to use for the select options.
   */
  public function resolve(array $element, ?int $depth = NULL): array {
    // Get the default properties from the plugin instance instead of using
    // hardcoded values, the defaults might have been altered by other modules.
    $default_properties = $this->elementManager->getElementInstance($element)->getDefaultProperties();

    // Get the vocabulary from the element definition, with the default value as
    // fallback.
    $vocabulary = $element['#vocabulary'] ?? $default_properties['vocabulary'] ?? NULL;
    if (empty($vocabulary)) {
      return [];
    }

    // Get the depth from the GraphQL field arguments, with the element
    // definition and the default value as fallbacks.
    $depth = $depth ?? $element['#depth'] ?? $default_properties['depth'] ?? NULL;
    $terms = $this->getTaxonomyTermStorage()->loadTree($vocabulary, 0, $depth, TRUE);

    $options = [];
    foreach ($terms as $term) {
      $this->cacheTags = array_merge($this->cacheTags, $term->getCacheTagsToInvalidate());
      $options[$term->id()] = [
        'value' => $term->id(),
        'title' => $term->getName(),
        'description' => $term->getDescription(),
      ];
    }

    return $options;
  }

  /**
   * Returns the term storage.
   *
   * @return \Drupal\taxonomy\TermStorageInterface
   *   The term storage.
   */
  protected function getTaxonomyTermStorage(): TermStorageInterface {
    return $this->entityTypeManager->getStorage('taxonomy_term');
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    return array_unique($this->cacheTags);
  }

}
