<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\Mutation;

use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\node\NodeInterface;

/**
 * Tests submission of a Webform in context of a source entity.
 *
 * @group graphql_webform
 */
class FormSubmissionWithSourceEntityTest extends FormSubmissionTestBase {

  /**
   * A test node serving as a source entity for a Webform submission.
   */
  protected NodeInterface $node;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('node_type');
    $this->installEntitySchema('node');

    // Create a test node.
    $node_type = NodeType::create(['type' => 'article', 'name' => 'Article']);
    $node_type->save();
    $this->node = Node::create(['type' => 'article', 'title' => 'Test node']);
    $this->node->save();
  }

  /**
   * Tests a form submission that includes a source entity type and ID.
   */
  public function testSubmissionWithSourceEntity(): void {
    $query = $this->getQueryFromFile('submission_source_entity.gql');
    $variables = [
      'id' => 'graphql_webform_test_form',
      'elements' => [(object) ['element' => 'required_text_field', 'value' => 'A value.']],
      'sourceEntityType' => 'node',
      'sourceEntityId' => $this->node->id(),
    ];

    $this->assertResults($query, $variables, [
      'submitWebform' => [
        'submission' => [
          'id' => '1',
          'webform' => [
            'id' => 'graphql_webform_test_form',
          ],
          'sourceEntityType' => 'node',
          'sourceEntityId' => (string) $this->node->id(),
        ],
      ],
    ], $this->defaultCacheMetaData()->addCacheTags(['webform_submission:1']));

    // There should be one submission in the database.
    $this->assertWebformSubmissionCount(1, 'A valid submission is saved.');

    // Load the webform submission and check it contains the expected reference.
    $submissions = $this->webformSubmissionStorage->loadMultiple();
    /** @var \Drupal\webform\WebformSubmissionInterface $submission */
    $submission = reset($submissions);
    $this->assertEquals('node', $submission->getSourceEntity()->getEntityTypeId());
    $this->assertEquals($this->node->id(), $submission->getSourceEntity()->id());
  }

}
