# GraphQL Webform Drupal module
A module to integrate webform with the graphql module.

IMPORTANT: This is a module under active development and it does not support all
webform features and elements yet. Feel free to raise Feature Requests and
to contribute :)

## Pre-Requisites

 1. [graphql module](https://www.drupal.org/project/graphql)
 2. [webform module](https://www.drupal.org/project/webform)

## Install

composer require drupal/graphql_webform

## Supported elements

 - Text Field
 - Email
 - Textarea
 - Hidden
 - Date
 - Checkboxes
 - Radios
 - Select
 - File
 - Actions
 - Markup
 - Webform Term Select
 - Webform Entity Select
 - Number
 - Composite (with the above items supported)
 - Time

## Retrieve webform elements

Make sure to grant the `access webform configuration` permission to the users
who need to access the webform over GraphQL. This can be done individually per
webform in Build > Settings > Access > Access Webform configuration, or globally
by granting the `access any webform configuration` permission.

```graphql
fragment metadata on WebformElementMetadata {
  key
  type
  title
  required {
    message
  }
}

fragment multiple on WebformElementMultipleValuesBase {
  multipleValues {
    limit
    message
  }
}

{
  webformById(id: "contact") {
    title
    description
    elements {
      ... on WebformElement {
        metadata {
          ...metadata
        }
      }
      ... on WebformElementWebformActions {
        submitLabel
      }
      ... on WebformElementTextBase {
        defaultValue
        size
        minlength
        maxlength
        pattern {
          message
          rule
        }
        placeholder
        multiple {
          limit  # 0 means Unlimited config.
          message
        }
      }
      ... on WebformElementMarkup {
        markup
      }
      ... on WebformElementTextarea {
        rows
      }
      ... on WebformElementHidden {
        defaultValue
      }
      ... on WebformElementDate {
        dateMin
        dateMax
        step
        defaultValue
        multiple {
          limit  # 0 means Unlimited config.
          message
        }
      }
      ... on WebformElementOptionsBase {
        defaultValue
        options {
          title
          value
        }
        multiple {
          limit  # 0 means Unlimited config.
          message
        }
      }
      ... on WebformElementSelect {
        emptyOption {
          title
          value
        }
      }
      ... on WebformElementManagedFile {
        fileExtensions
        multiple {
          limit  # 0 means Unlimited config.
          message
        }
      }
      ... on WebformElementWebformTermSelect {
        metadata {
          ...metadata
        }
        options(depth: 1) {
          value
          title
        }
      }
      ... on WebformElementWebformCustomComposite {
        elements {
          metadata {
            ... metadata
          }
          ... multiple
        }
      }
      ... on WebformElementNumber{
        min
        max
        size
        step
      }
    }
  }
}
```

## Create a webform submission

    mutation submit(
      $id: String!,
      $elements: [WebformSubmissionElement]!,
    ) {
      submitWebform(
        id: $id,
        elements: $elements,
      ) {
        errors
        validationErrors {
          element
          messages
        }
        submission {
          id
          webform {
            id
          }
        }
      }
    }

The `$elements` argument is an array of `WebformSubmissionElement` objects. Here
is an example payload:

    {
      "id": "my_webform",
      "elements": [
        {
          "element": "email",
          "value": "me@mysite.com"
        },
        {
          "element": "checkboxes",
          "value": [
            "some_value",
            "another_value"
          ]
        }
      ]
    }

`validationErrors` will contain error messages for each element that failed
validation. `errors` will contain general errors, such as if the webform
submission failed because the form is closed.


### Create a webform submission when webform contains File elements
If the webform contains managed file fields, you can attach the uploaded files
by using a multipart form request and including a 'map' field that maps the
uploaded files to the webform elements.

For more information on how to upload files using GraphQL, see the
[GraphQL multipart request specification](https://github.com/jaydenseric/graphql-multipart-request-spec).

```
mutation submitMyForm(
  $id: String!,
  $elements: [WebformSubmissionElement]!,
  $files: [WebformSubmissionFile]!,
) {
  submitWebform(
    id: $id,
    elements: $elements,
    files: $files,
  ) {
    errors
    validationErrors {
      element
      messages
    }
    submission {
      id
    }
  }
}
```

Example CURL request. Note that the value of the `file` field is `null` in the
`variables` object. It will be replaced by the file linked in the `map` object.

```
curl localhost:3001/graphql \
  -F operations='{
    "query": "mutation ($id: String!, $files: [WebformSubmissionFile]!) { submitWebform(id: $id, files: $files) { errors validationErrors { element messages } submission { id } } }",
    "variables": {
      "id": "my_webform",
      "files": [{"element": "my_file_element", "file": null}]
    }
  }' \
  -F map='{"0": ["variables.files.0.file"]}' \
  -F 0=@a.txt
```

### Create a webform submission with a source entity
If the webform supports a source entity, you can pass in the variables with the
data, the same as with `id`. For example, if your webform's URL is:
`/form/my-form?source_entity_id=123&source_entity_type=node`

Pass the following JSON encoded values:

```
{
    "id": "my_form",
    "source_entity_id": "123",
    "source_entity_type": "node",
    "subject":"This is the subject",
    "message":"Hey, I have a question",
    "date_of_birth":"05/01/1930",
    "email":"email@example.com"
  }
```

The mutation will automatically pass these values to the webform submission.

### Working with composite elements and containers

Composite elements and containers are both elements that contain other elements.
They can be configured to have multiple values, and also the child elements can
accept multiple values. Since the Webform module does not know in advance how
many elements will be submitted, the element keys do not include the delta.

To retrieve the element structure of a webform containing composite elements,
you can use a query like the following:

```graphql
fragment metadata on WebformElement {
  metadata {
    key
    type
    title
  }
}

fragment multiple on WebformElementMultipleValuesBase {
  multipleValues {
    limit
  }
}

query getWebformElements($id: String!) {
  webformById(id: $id) {
    elements {
      ... on WebformElementWebformCustomComposite {
        ...metadata
        ...multiple
        elements {
          ...metadata
          ...multiple
        }
      }
    }
  }
}
```

This will return data similar to this:
```json
{
  "data": {
    "webformById": {
      "elements": [
        {
          "metadata": {
            "key": "custom_composite_container",
            "type": "webform_custom_composite",
            "title": "Custom composite container"
          },
          "multipleValues": {
            // This indicates that the container can have unlimited values.
            "limit": -1
          },
          "elements": [
            {
              "metadata": {
                // Note that the element keys do not include the delta.
                "key": "custom_composite_container__first_name",
                "type": "textfield",
                "title": "First name"
              },
              "multipleValues": null
            },
            {
              "metadata": {
                "key": "custom_composite_container__last_name",
                "type": "textfield",
                "title": "Last name"
              },
              "multipleValues": null
            }
          ]
        }
      ]
    }
  }
}
```

The element keys are in the format `container_key__child_key`. When submitting
the form, you should take into account whether the elements accept multiple
values, and name the keys accordingly.

If the container only accepts a single value, the delta should not be included,
so the submission should look like this:

```json
{
  "id": "my_webform",
  "elements": [
    {
      "element": "custom_composite_container[first_name]",
      "value": "John"
    },
    {
      "element": "custom_composite_container[last_name]",
      "value": "Doe"
    }
  ]
}
```

While if the container accepts multiple values, the submission should include
the delta for each submitted entry:

```json
{
  "id": "my_webform",
  "elements": [
    {
      "element": "custom_composite_container[0][first_name]",
      "value": "John"
    },
    {
      "element": "custom_composite_container[0][last_name]",
      "value": "Doe"
    },
    {
      "element": "custom_composite_container[1][first_name]",
      "value": "Jane"
    },
    {
      "element": "custom_composite_container[1][last_name]",
      "value": "Doe"
    }
  ]
}
```
