<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\Utility\WebformOptionsHelper;

/**
 * Returns the options of a Webform element.
 *
 * @DataProducer(
 *   id = "webform_element_options",
 *   name = @Translation("Webform element options"),
 *   description = @Translation("Returns the available options for a Webform element. Used for checkboxes, radio buttons, dropdowns etc."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Any value"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "element" = @ContextDefinition(
 *       "any",
 *       label = @Translation("Webform element")
 *     )
 *   }
 * )
 */
class WebformElementOptions extends DataProducerPluginBase {

  /**
   * Resolves options for a Webform element.
   *
   * @param array $element
   *   The webform element from which to return the options.
   *
   * @return \Iterator
   *   The available options.
   */
  public function resolve(array $element): \Iterator {
    // Don't return options for disabled elements.
    if ($element['#disabled'] ?? FALSE) {
      return [];
    }

    $options = $element['#options'] ?? [];
    // Strip out the empty option. This is exposed separately as the emptyOption
    // field.
    if (isset($element['#empty_value'])) {
      unset($options[$element['#empty_value']]);
    }

    foreach ($options as $value => $data) {
      $parts = WebformOptionsHelper::splitOption($data);
      yield [
        'value' => $value,
        'title' => $parts[0],
        'description' => $parts[1] ?? '',
      ];
    }
  }

}
