<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Render\RendererInterface;
use Drupal\file\FileInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\GraphQL\Utility\FileUpload;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\graphql_webform\Model\WebformSubmissionResult;
use Drupal\webform\Element\Webform;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Drupal\webform\WebformEntityStorageInterface;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformSubmissionForm;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Produces a Webform submission result containing the input data.
 *
 * @DataProducer(
 *   id = "webform_submit",
 *   name = @Translation("Webform submit"),
 *   description = @Translation("Create a webform submission."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Webform submission result"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "id" = @ContextDefinition("string",
 *       label = @Translation("Webform ID")
 *     ),
 *     "elements" = @ContextDefinition("any",
 *       label = @Translation("Array of WebformSubmissionElement objects."),
 *       multiple = TRUE
 *     ),
 *     "files" = @ContextDefinition("any",
 *       label = @Translation("Array of WebformSubmissionFile objects."),
 *       multiple = TRUE
 *     ),
 *     "sourceEntityType" = @ContextDefinition("string",
 *       label = @Translation("Source entity type"),
 *       required = FALSE
 *     ),
 *     "sourceEntityId" = @ContextDefinition("string",
 *       label = @Translation("Source entity ID"),
 *       required = FALSE
 *     ),
 *   }
 * )
 */
class WebformSubmit extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new WebformSubmit data producer.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin ID.
   * @param mixed $pluginDefinition
   *   The plugin definition.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\graphql\GraphQL\Utility\FileUpload $fileUpload
   *   GraphQL file upload service.
   * @param \Drupal\webform\Plugin\WebformElementManagerInterface $elementManager
   *   The Webform element manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected readonly RendererInterface $renderer,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly FileUpload $fileUpload,
    protected readonly WebformElementManagerInterface $elementManager,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $pluginId, $pluginDefinition) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('renderer'),
      $container->get('entity_type.manager'),
      $container->get('graphql.file_upload'),
      $container->get('plugin.manager.webform.element'),
      $container->get('config.factory')
    );
  }

  /**
   * Resolves the Webform submission.
   *
   * @param string $id
   *   The Webform ID for which to create a submission.
   * @param array $elements
   *   Submitted values for Webform elements.
   * @param array $files
   *   Submitted file uploads.
   * @param string|null $sourceEntityType
   *   The entity type of the entity that is the source of the submission.
   * @param string|null $sourceEntityId
   *   The ID of the entity that is the source of the submission.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $field
   *   The GraphQL field context.
   *
   * @return \Drupal\graphql_webform\Model\WebformSubmissionResult
   *   The Webform submission result.
   */
  public function resolve(
    string $id,
    array $elements,
    array $files,
    ?string $sourceEntityType,
    ?string $sourceEntityId,
    FieldContext $field,
  ): WebformSubmissionResult {
    // Create context and execute validation and submission in render context to
    // capture caching metadata.
    $renderContext = new RenderContext();
    $renderResult = $this->renderer->executeInRenderContext($renderContext, function () use ($id, $elements, $files, $field, $sourceEntityId, $sourceEntityType) {
      // Create the result object.
      $result = new WebformSubmissionResult();

      // Return an error if the requested Webform does not exist.
      $webform = $this->getWebformEntityStorage()->load($id);
      if (!$webform instanceof WebformInterface) {
        return $result->addError(sprintf('Webform %s does not exist.', $id));
      }

      $field->addCacheableDependency($webform);

      // Return an error if the Webform does not accept submissions.
      if (!$webform->isOpen()) {
        // Return the custom message defined in the form settings, falling back
        // to the default message. If the default message is empty, return a
        // helpful message.
        $message = $webform->getSetting('form_close_message');
        if (empty($message)) {
          $message = $this->getWebformSetting('settings.default_form_close_message');
        }
        if (empty($message)) {
          $message = 'This form is closed to new submissions.';
        }

        return $result->addError($message);
      }

      // Return an error if user is not allowed to create submissions.
      if (!$webform->access('submission_create')) {
        $accessDenied = Webform::buildAccessDenied($webform);
        $markup = (string) $this->renderer->render($accessDenied);
        return $result->addError($markup);
      }

      // Convert the elements array into an associative array keyed by element
      // name, so it matches the data structure expected by Drupal's form
      // submission handlers.
      $dataElements = [];
      foreach ($elements as $element) {
        $dataElements[$element['element']] = $element['value'];
      }

      // Handle file uploads. This will add validation errors if necessary.
      $createdFileIds = $this->handleFileElements($webform, $files, $result);
      foreach ($createdFileIds as $elementKey => $fileIds) {
        $dataElements[$elementKey] = $fileIds;
      }

      // We are passed in a flat key-value array, expand this into a nested
      // array that aligns with the way values for form submissions with nested
      // elements are handled in the Form API.
      $dataElements = $this->unflattenElementKeys($dataElements);

      // Build the form data object for the form submission.
      $formData = [
        'webform_id' => $id,
        'data' => $dataElements,
      ];

      // Only include the source entity type and ID if both are provided.
      if ($sourceEntityId && $sourceEntityType) {
        $formData['entity_id'] = $sourceEntityId;
        $formData['entity_type'] = $sourceEntityType;
      }

      // Let the Webform module handle the validation of the form values.
      $validationErrors = WebformSubmissionForm::validateFormValues($formData) ?? [];
      $this->handleFormErrors($validationErrors, $result);

      // Check if everything is valid so far. Do not create a submission if
      // there are any validation errors.
      if (!$result->isValid()) {
        return $result;
      }

      // Try to create a submission.
      $submission = WebformSubmissionForm::submitFormValues($formData);

      if ($submission instanceof WebformSubmissionInterface) {
        // The submission was created successfully. Return it.
        $result->setSubmission($submission);
      }
      elseif (!empty($submission)) {
        // An error happened during submission creation.
        $this->handleFormErrors($submission, $result);
      }

      return $result;
    });

    if (!$renderContext->isEmpty()) {
      $field->addCacheableDependency($renderContext->pop());
    }

    return $renderResult;
  }

  /**
   * Populates errors on the result as WebformSubmissionValidationError objects.
   *
   * @param array $errors
   *   The errors returned by the Webform module.
   * @param \Drupal\graphql_webform\Model\WebformSubmissionResult $result
   *   The result object to populate.
   */
  protected static function handleFormErrors(array $errors, WebformSubmissionResult $result): void {
    foreach ($errors as $elementId => $message) {
      // The Webform module returns error messages intended to be shown in a web
      // page. Strip HTML tags to make them suitable for general use.
      $message = strip_tags((string) $message);
      $result->addValidationError($message, $elementId);
    }
  }

  /**
   * Handles possible file uploads.
   *
   * @param \Drupal\webform\WebformInterface $webform
   *   The Webform entity.
   * @param array $uploadedFiles
   *   The array of files that were uploaded in the form submission.
   * @param \Drupal\graphql_webform\Model\WebformSubmissionResult $result
   *   The result object to populate with validation errors.
   *
   * @return array
   *   An array of successfully uploaded file IDs, grouped by element key.
   */
  protected function handleFileElements(WebformInterface $webform, array $uploadedFiles, WebformSubmissionResult $result): array {
    // Group the files by element so we can process each element separately.
    $filesByElement = [];
    foreach ($uploadedFiles as $file) {
      $filesByElement[$file['element']][] = $file['file'];
    }

    // Retrieve all the managed file elements so we can check if the uploaded
    // files are correctly associated with a managed file element.
    $managedFileElements = array_keys($webform->getElementsManagedFiles());

    // Keep track of File entity IDs that were created by valid uploads. This
    // will be returned at the end of the method.
    $fileIds = [];

    foreach ($filesByElement as $elementKey => $files) {
      // Check if the element is actually a managed file element. If not, report
      // this as a generic error rather than a validation error. This is
      // probably not the end user's mistake but rather a bug in the GraphQL
      // client code.
      if (!in_array($elementKey, $managedFileElements)) {
        $error = sprintf('Files cannot be uploaded to the "%s" element since it is not a managed file element.', $elementKey);
        $result->addError($error);
        continue;
      }

      // Get the initialized render element and the plugin instance that
      // represents an OOP interface to the element configuration.
      $element = $webform->getElement($elementKey);
      /** @var \Drupal\webform\Plugin\WebformElement\WebformManagedFileBase $instance */
      $instance = $this->elementManager->getElementInstance($element);
      $instance->prepare($element);

      // Get the number of allowed files for this element.
      $allowedNumberOfFiles = $instance->hasMultipleValues($element);

      // Don't check the number of uploaded files if an unlimited number of
      // files is allowed.
      if ($allowedNumberOfFiles !== TRUE) {
        $allowedNumberOfFiles = is_numeric($allowedNumberOfFiles) ? (int) $allowedNumberOfFiles : 1;
        if (count($files) > $allowedNumberOfFiles) {
          $message = match ($allowedNumberOfFiles) {
            1 => 'Only one file can be uploaded.',
            default => sprintf('The number of files uploaded exceeds the maximum of %d.', $allowedNumberOfFiles),
          };
          $result->addValidationError($message, $elementKey);
          continue;
        }
      }

      // Retrieve the validation criteria for the file upload element. Note that
      // the methods to retrieve the file size and allowed extensions are
      // protected, so we need to get them from the render array instead.
      // @see \Drupal\webform\Plugin\WebformElement\WebformManagedFileBase::getFileExtensions()
      // @see \Drupal\webform\Plugin\WebformElement\WebformManagedFileBase::getMaxFileSize()
      $uriScheme = $instance->getElementProperty($element, 'uri_scheme');
      $maxFilesize = $element['#upload_validators']['FileSizeLimit']['fileLimit'];
      $fileExtensions = $element['#upload_validators']['FileExtension']['extensions'];

      // Attempt to save the uploaded files, leveraging the validation checks of
      // the file upload service from the GraphQL module.
      foreach ($files as $file) {
        $uploadResponse = $this->fileUpload->saveFileUpload($file, [
          'file_extensions' => $fileExtensions,
          'max_filesize' => $maxFilesize,
          'uri_scheme' => $uriScheme,
          'file_directory' => 'webform',
        ]);

        // If there are any violations, add them as form validation errors.
        $violations = $uploadResponse->getViolations();
        if (!empty($violations)) {
          foreach ($violations as $violation) {
            $result->addValidationError($violation['message'], $elementKey);
          }
          continue;
        }

        $file = $uploadResponse->getFileEntity();
        // At this point we should have a file. If we don't, inform the client
        // that the file was not uploaded.
        // @todo If this ever happens in practice, we should probably log this.
        if (!$file instanceof FileInterface) {
          $result->addValidationError('Unexpected error occurred while uploading file. Please try again later.', $elementKey);
          continue;
        }

        // The webform module expects that values submitted for multi-value
        // elements are passed as arrays, while values for single value elements
        // are passed as a scalar. Avoid array to string conversion errors by
        // casting the file ID accordingly.
        // @see \Drupal\webform\WebformSubmissionStorage::saveData()
        if ($allowedNumberOfFiles === 1) {
          $fileIds[$elementKey] = $file->id();
        }
        else {
          $fileIds[$elementKey][] = $file->id();
        }
      }
    }

    return $fileIds;
  }

  /**
   * Transform an array with flattened element keys into a nested array.
   *
   * To keep frontend implementations simple the submission values are provided
   * as an array of elements with their input name and value as keys. The input
   * array might look like this:
   *
   * @code
   * [
   *   "message" => "Hello",
   *   "users[0][name]" => "John",
   *   "users[0][email]" => "foobar@domain.com",
   *   "users[1][name]" => "Wayne",
   *   "users[1][email]" => "baz@domain.com",
   * ]
   * @endcode
   *
   * This method converts this into a nested array which can be used to create a
   * webform submission:
   *
   * @code
   * [
   *   "message" => "Hello",
   *   "users" => [
   *     [
   *       "name" => "John",
   *       "email" => "foobar@domain.com",
   *     ],
   *     [
   *       "name" => "Wayne",
   *       "email" => "baz@domain.com",
   *     ]
   *   ],
   * ]
   * @endcode
   *
   * @param array $values
   *   The array of values to unflatten by element key.
   *
   * @return array
   *   The values converted into a nested format.
   */
  protected function unflattenElementKeys(array $values): array {
    $result = [];

    foreach ($values as $elementKey => $value) {
      // If it is a nested key, unflatten it into a nested array.
      if (str_contains($elementKey, '[')) {
        // Split the key into its parts by exploding it at the brackets.
        $keys = preg_split('/[\[\]]+/', $elementKey, -1, PREG_SPLIT_NO_EMPTY);

        // Start at the root of the result array.
        $temp = &$result;

        // Loop over all the keys, creating nested arrays as necessary.
        foreach ($keys as $key) {
          if (!isset($temp[$key])) {
            $temp[$key] = [];
          }
          // Move one level deeper so we can process the next key.
          $temp = &$temp[$key];
        }

        // We reached the deepest level, now assign the original value to this
        // level.
        $temp = $value;
      }
      // If it is not a nested key, just assign the value.
      else {
        $result[$elementKey] = $value;
      }
    }

    return $result;
  }

  /**
   * Returns the webform entity storage.
   *
   * @return \Drupal\webform\WebformEntityStorageInterface
   *   The webform entity storage.
   */
  protected function getWebformEntityStorage(): WebformEntityStorageInterface {
    return $this->entityTypeManager->getStorage('webform');
  }

  /**
   * Returns the setting from the Webform module with the given key.
   *
   * @param string $key
   *   The key of the setting to return.
   *
   * @return mixed
   *   The setting value.
   */
  protected function getWebformSetting(string $key) {
    return $this->configFactory->get('webform.settings')->get($key);
  }

}
