<?php

declare(strict_types=1);

namespace Drupal\graphql_webform;

use Drupal\graphql_webform\Enum\WebformElementDisplayOn;
use Drupal\webform\Plugin\WebformElement\ContainerBase;
use Drupal\webform\Plugin\WebformElement\OptionsBase;
use Drupal\webform\Plugin\WebformElement\TextBase;
use Drupal\webform\Plugin\WebformElement\WebformManagedFileBase;
use Drupal\webform\Plugin\WebformElement\WebformMarkupBase;
use Drupal\webform\Plugin\WebformElement\WebformTermSelect;
use Drupal\webform\Plugin\WebformElementInterface;
use GraphQL\Type\Definition\EnumType;
use GraphQL\Type\Definition\InterfaceType;
use GraphQL\Type\Definition\ObjectType;
use GraphQL\Type\Definition\Type;
use GraphQL\Type\Schema;
use GraphQL\Utils\SchemaPrinter;

/**
 * Builds the GraphQL schema for the Webform module.
 */
class WebformSchemaBuilder {

  /**
   * The generated types.
   *
   * @var \GraphQL\Type\Definition\TypeWithFields[]
   */
  protected $types;

  /**
   * The WebformElementContainerBase interface fields.
   */
  protected array $elementContainerBase;

  /**
   * The WebformElementManagedFileBase interface fields.
   */
  protected array $elementManagedFileBase;

  /**
   * The WebformElementTextBase interface fields.
   */
  protected array $elementTextBase;

  public function __construct() {
    $this->types = [];

    $this->types['WebformRequiredElement'] = new ObjectType([
      'name' => 'WebformRequiredElement',
      'description' => 'Indicates that the Webform element is required.',
      'fields' => [
        'message' => $this->type(Type::string(), 'The error message to display when the element is required but empty.'),
      ],
    ]);

    $this->types['WebformElementMetadata'] = new ObjectType([
      'name' => 'WebformElementMetadata',
      'fields' => [
        'key' => $this->type(Type::nonNull(Type::string()), 'The key of the element.'),
        'type' => $this->type(Type::nonNull(Type::string()), 'The element type.'),
        'title' => $this->type(Type::string(), ''),
        'description' => $this->type(Type::string(), ''),
        'multiple' => $this->type(Type::int(), 'The number of values that can be entered into the element. Unlimited if 0.'),
        'required' => $this->types['WebformRequiredElement'],
      ],
    ]);

    $this->types['WebformElement'] = new InterfaceType([
      'name' => 'WebformElement',
      'fields' => $this->getElementDefinition(),
    ]);

    $this->elementContainerBase = [
      'elements' => fn () => Type::nonNull(Type::listOf(Type::nonNull($this->types['WebformElement']))),
    ];

    $this->types['WebformElementContainerBase'] = new InterfaceType([
      'name' => 'WebformElementContainerBase',
      'fields' => $this->elementContainerBase,
    ]);

    $this->types['WebformElementMultipleValues'] = new ObjectType([
      'name' => 'WebformElementMultipleValues',
      'fields' => [
        'limit' => [
          'type' => Type::int(),
          'description' => 'The maximum number of values that can be entered. If set to -1, an unlimited number of values can be entered.',
        ],
        'message' => [
          'type' => Type::string(),
          'description' => 'The error message to display when the maximum number of values is exceeded.',
        ],
        'headerLabel' => [
          'type' => Type::string(),
          'description' => 'The label for the header of the multiple values fieldset.',
        ],
        'minItems' => [
          'type' => Type::int(),
          'description' => 'The minimum number of items that must be entered.',
        ],
        'emptyItems' => [
          'type' => Type::int(),
          'description' => 'The number of empty items to display.',
        ],
        'addMore' => [
          'type' => Type::boolean(),
          'description' => 'Whether or not to display the "Add more" button.',
        ],
        'addMoreItems' => [
          'type' => Type::int(),
          'description' => 'The number of items to add when the "Add more" button is clicked.',
        ],
        'addMoreButtonLabel' => [
          'type' => Type::string(),
          'description' => 'The label for the "Add more" button.',
        ],
        'addMoreInput' => [
          'type' => Type::boolean(),
          'description' => 'Allow users to input the number of items to be added.',
        ],
        'addMoreInputLabel' => [
          'type' => Type::string(),
          'description' => 'The label for the input field that allows users to input the number of items to be added.',
        ],
        'itemLabel' => [
          'type' => Type::string(),
          'description' => 'The label for each item.',
        ],
        'noItemsMessage' => [
          'type' => Type::string(),
          'description' => 'The message to display when there are no items.',
        ],
        'sorting' => [
          'type' => Type::boolean(),
          'description' => 'Allow users to sort elements.',
        ],
        'operations' => [
          'type' => Type::boolean(),
          'description' => 'Allow users to add/remove elements.',
        ],
        'add' => [
          'type' => Type::boolean(),
          'description' => 'Whether to show the "Add" button.',
        ],
        'remove' => [
          'type' => Type::boolean(),
          'description' => 'Whether to show the "Remove" button.',
        ],
      ],
    ]);

    $this->types['WebformElementMultipleValuesBase'] = new InterfaceType([
      'name' => 'WebformElementMultipleValuesBase',
      'fields' => [
        'multipleValues' => $this->types['WebformElementMultipleValues'],
      ],
    ]);

    $this->types['WebformElementOption'] = new ObjectType([
      'name' => 'WebformElementOption',
      'fields' => [
        'value' => Type::nonNull(Type::string()),
        'description' => Type::string(),
        'title' => Type::nonNull(Type::string()),
      ],
    ]);

    $this->types['WebformElementOptionsBase'] = new InterfaceType([
      'name' => 'WebformElementOptionsBase',
      'fields' => [
        'options' => fn () => Type::listOf($this->types['WebformElementOption']),
      ],
    ]);

    $this->types['WebformElementDisplayOn'] = new EnumType([
      'name' => 'WebformElementDisplayOn',
      'values' => WebformElementDisplayOn::toArray(),
      'description' => 'The `WebformElementDisplayOn` Enum contains the various locations where the associated Webform element can be displayed: during form editing, form viewing, both, or not visible at all.',
    ]);

    $this->types['WebformElementMarkupBase'] = new InterfaceType([
      'name' => 'WebformElementMarkupBase',
      'fields' => $this->getMarkupBaseFields(),
    ]);

    $this->elementManagedFileBase = [
      'maxFilesize' => Type::int(),
      'fileExtensions' => Type::string(),
    ];

    $this->types['WebformElementManagedFileBase'] = new InterfaceType([
      'name' => 'WebformElementManagedFileBase',
      'fields' => $this->elementManagedFileBase,
    ]);

    // A type representing a Regex validation pattern with an error message.
    $this->types['WebformElementValidationPattern'] = new ObjectType([
      'name' => 'WebformElementValidationPattern',
      'fields' => [
        'rule' => Type::string(),
        'message' => Type::string(),
      ],
    ]);

    $this->elementTextBase = [
      'readonly' => Type::boolean(),
      'size' => Type::int(),
      'minlength' => Type::int(),
      'maxlength' => Type::int(),
      'placeholder' => Type::string(),
      'autocomplete' => Type::string(),
      'pattern' => $this->types['WebformElementValidationPattern'],
    ];

    $this->types['WebformElementTextBase'] = new InterfaceType([
      'name' => 'WebformElementTextBase',
      'fields' => $this->elementTextBase,
    ]);
  }

  /**
   * Returns the GraphQL schema describing the Webform data types.
   *
   * @return string
   *   The generated schema.
   */
  public function getGeneratedSchema(): string {
    $schema = new Schema(['types' => $this->types]);

    return SchemaPrinter::doPrint($schema);
  }

  /**
   * Generate the types for the form elements.
   */
  public function generateElementType(string $id, string $description, ?WebformElementInterface $plugin = NULL): void {
    $interfaces = [fn () => $this->types['WebformElement']];
    $fields = $this->getElementDefinition();

    // Expose child elements on elements that have them.
    $elementIsContainer =
      // Containers: fieldsets, flexbox elements, etc.
      $plugin instanceof ContainerBase ||
      // Elements that are composed of multiple fields, e.g. address fields.
      $plugin?->isComposite();

    if ($elementIsContainer) {
      $interfaces[] = fn () => $this->types['WebformElementContainerBase'];
      $fields = [
        ...$fields,
        ...$this->elementContainerBase,
      ];
    }

    if ($plugin instanceof OptionsBase) {
      $interfaces[] = fn () => $this->types['WebformElementOptionsBase'];
      $fields = [
        ...$fields,
        'options' => fn () => Type::listOf($this->types['WebformElementOption']),
      ];
    }

    if ($plugin instanceof WebformElementInterface && $plugin->supportsMultipleValues()) {
      $interfaces[] = fn () => $this->types['WebformElementMultipleValuesBase'];
      $fields = [
        ...$fields,
        'multipleValues' => fn () => $this->types['WebformElementMultipleValues'],
      ];
    }

    // The Webform element for selecting taxonomy terms has an option to limit
    // the depth of terms to choose from. Allow this to be overridden in the
    // GraphQL query.
    if ($plugin instanceof WebformTermSelect) {
      $fields['options'] = [
        'type' => Type::listOf($this->types['WebformElementOption']),
        'description' => 'The taxonomy terms to show in the select element.',
        'args' => [
          'depth' => [
            'type' => Type::int(),
            'description' => 'The depth of terms to show in the select element. If omitted this will use the depth set in the Webform element configuration.',
          ],
        ],
      ];
    }

    if ($plugin instanceof WebformMarkupBase) {
      $interfaces[] = fn () => $this->types['WebformElementMarkupBase'];
      $fields = array_merge($fields, $this->getMarkupBaseFields());
    }

    if ($plugin instanceof WebformManagedFileBase) {
      $interfaces[] = fn () => $this->types['WebformElementManagedFileBase'];
      $fields = array_merge($fields, $this->elementManagedFileBase);
    }

    if ($plugin instanceof TextBase) {
      $interfaces[] = fn () => $this->types['WebformElementTextBase'];
      $fields = [
        ...$fields,
        ...$this->elementTextBase,
      ];
    }

    $type_name = 'WebformElement' . $this->toUpperCamelCase($id);
    $this->types[$type_name] = new ObjectType([
      'name' => $type_name,
      'fields' => $fields,
      'description' => $description,
      'interfaces' => $interfaces,
    ]);
  }

  /**
   * Formats the given type and description as an array.
   *
   * @param \GraphQL\Type\Definition\Type $type
   *   The GraphQL type.
   * @param string $description
   *   The description.
   *
   * @return array
   *   An array with the type and description.
   */
  protected static function type(Type $type, string $description): array {
    return [
      'type' => $type,
      'description' => $description,
    ];
  }

  /**
   * Returns the fields that are common to all markup elements.
   *
   * @return array
   *   The 'markup' and 'displayOn' fields.
   */
  protected function getMarkupBaseFields(): array {
    return [
      'markup' => Type::string(),
      'displayOn' => $this->types['WebformElementDisplayOn'],
    ];
  }

  /**
   * Returns the definition for a GraphQL element.
   *
   * @return array
   *   The element definition, containing the base fields.
   */
  protected function getElementDefinition(): array {
    return [
      'metadata' => fn () => $this->types['WebformElementMetadata'],
    ];
  }

  /**
   * Converts the given string to upper camel case.
   *
   * Dashes and underscores are removed.
   *
   * @param string $string
   *   The string to convert.
   *
   * @return string
   *   The converted string.
   */
  protected function toUpperCamelCase(string $string): string {
    return str_replace(' ', '', ucwords(str_replace(['-', '_'], ' ', $string)));
  }

}
