<?php

declare(strict_types=1);

namespace Drupal\Tests\graphql_webform\Kernel\DataProducer;

use Drupal\KernelTests\KernelTestBase;
use Drupal\graphql_webform\Plugin\GraphQL\DataProducer\WebformElementProperty;

/**
 * Tests for the webform element property data producer.
 *
 * @coversDefaultClass \Drupal\graphql_webform\Plugin\GraphQL\DataProducer\WebformElementProperty
 * @group graphql_webform
 */
class WebformElementPropertyTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'file',
    'graphql',
    'graphql_webform',
    'webform',
    'user',
  ];

  /**
   * The webform element property data producer. This is the system under test.
   */
  protected WebformElementProperty $webformElementProperty;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $plugin_manager = $this->container->get('plugin.manager.graphql.data_producer');
    $this->webformElementProperty = $plugin_manager->createInstance('webform_element_property');
  }

  /**
   * Tests resolving of webform element properties.
   *
   * @covers ::resolve
   * @dataProvider resolveDataProvider
   */
  public function testResolve(array $element, string $property, string $type, mixed $expected): void {
    $actual = $this->webformElementProperty->resolve($element, $property, $type);
    self::assertSame($expected, $actual);
  }

  /**
   * Provides test data for testResolve().
   */
  public static function resolveDataProvider(): array {
    return [
      'simple string property' => [
        [
          '#type' => 'textfield',
          '#title' => 'Title',
        ],
        'title',
        'string',
        'Title',
      ],
      'description property' => [
        [
          '#type' => 'textfield',
          '#description' => [
            '#theme' => 'webform_html_editor_markup',
            '#markup' => 'Description',
          ],
        ],
        'description',
        'string',
        'Description',
      ],
      'boolean property' => [
        [
          '#type' => 'textfield',
          '#required' => TRUE,
        ],
        'required',
        'boolean',
        TRUE,
      ],
      'array property' => [
        [
          '#type' => 'textfield',
          '#attributes' => [
            'class' => ['class1', 'class2'],
          ],
        ],
        'attributes',
        'array',
        [
          'class' => ['class1', 'class2'],
        ],
      ],
      'non-existing property' => [
        [
          '#type' => 'textfield',
        ],
        'non_existing',
        'string',
        NULL,
      ],
    ];
  }

}
