<?php

declare(strict_types=1);

namespace Drupal\graphql_webform;

/**
 * Converts strings to different cases.
 */
class CaseConverter {

  /**
   * Converts the given string to UpperCamelCase.
   *
   * Dashes and underscores are removed.
   *
   * @param string $string
   *   The string to convert.
   *
   * @return string
   *   The converted string.
   */
  public static function toUpperCamelCase(string $string): string {
    return str_replace(' ', '', ucwords(str_replace(['-', '_'], ' ', $string)));
  }

  /**
   * Converts the given string to camelCase.
   *
   * Dashes and underscores are removed.
   *
   * @param string $string
   *   The string to convert.
   *
   * @return string
   *   The converted string.
   */
  public static function toCamelCase(string $string): string {
    return lcfirst(static::toUpperCamelCase($string));
  }

  /**
   * Converts the given string to snake_case.
   *
   * @param string $string
   *   The input string.
   *
   * @return string
   *   The converted string.
   */
  public static function toSnakeCase(string $string): string {
    // Replace non-letter/digit characters with underscores.
    $string = preg_replace('/[^a-zA-Z0-9]+/', '_', $string);

    // Add underscore before any uppercase letter preceded by a lowercase or
    // digit.
    $string = preg_replace('/([a-z0-9])([A-Z])/', '$1_$2', $string);

    // Convert to lowercase.
    $string = strtolower($string);

    // Trim underscores from start and end.
    return trim($string, '_');
  }

  /**
   * Converts the given string to SCREAMING_SNAKE_CASE.
   *
   * @param string $string
   *   The input string.
   *
   * @return string
   *   The converted string.
   */
  public static function toScreamingSnakeCase(string $string): string {
    return strtoupper(static::toSnakeCase($string));
  }

  /**
   * Converts the given string to kebab-case.
   *
   * @param string $string
   *   The input string.
   *
   * @return string
   *   The converted string.
   */
  public static function toKebabCase(string $string): string {
    return str_replace('_', '-', static::toSnakeCase($string));
  }

}
