<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Model;

use Drupal\webform\WebformSubmissionInterface;

/**
 * Data object representing the result of a Webform submission.
 */
class WebformSubmissionResult {

  /**
   * The Webform submission entity, if the submission was successful.
   */
  protected ?WebformSubmissionInterface $submission = NULL;

  /**
   * A list of error messages that occurred during submission.
   *
   * @var string[]
   */
  protected array $errors = [];

  /**
   * A list of validation errors that occurred during submission.
   *
   * @var \Drupal\graphql_webform\Model\WebformSubmissionValidationError[]
   */
  protected array $validationErrors = [];

  /**
   * Sets the Webform submission.
   *
   * @param \Drupal\webform\WebformSubmissionInterface $submission
   *   The Webform submission entity.
   *
   * @return \Drupal\graphql_webform\Model\WebformSubmissionResult
   *   The current class, for chaining.
   */
  public function setSubmission(WebformSubmissionInterface $submission): static {
    $this->submission = $submission;
    return $this;
  }

  /**
   * Returns the Webform submission entity.
   *
   * @return \Drupal\webform\WebformSubmissionInterface|null
   *   The Webform submission entity.
   */
  public function getSubmission(): ?WebformSubmissionInterface {
    return $this->submission;
  }

  /**
   * Adds an error message.
   *
   * These are intended for general errors. For validation errors, call
   * ::addValidationError() instead.
   *
   * @param string $error
   *   The error to add.
   *
   * @return \Drupal\graphql_webform\Model\WebformSubmissionResult
   *   The WebformSubmissionResult, for chaining.
   */
  public function addError(string $error): static {
    // The Webform module returns error messages intended to be shown in a web
    // page. Strip HTML tags to make them suitable for general use.
    $error = trim(strip_tags($error));
    $this->errors[] = $error;
    return $this;
  }

  /**
   * Returns the list of errors.
   *
   * @return string[]
   *   The errors.
   */
  public function getErrors(): array {
    return $this->errors;
  }

  /**
   * Returns the list of validation errors.
   *
   * @return \Drupal\graphql_webform\Model\WebformSubmissionValidationError[]
   *   The validation errors.
   */
  public function getValidationErrors(): array {
    return $this->validationErrors;
  }

  /**
   * Returns the validation error that corresponds to the given element.
   *
   * @param string|null $elementId
   *   The ID of the Webform element for which to return the validation error,
   *   or NULL to return the general validation error.
   *
   * @return \Drupal\graphql_webform\Model\WebformSubmissionValidationError
   *   The validation error. If no error exists for the given element ID yet, an
   *   empty error object is returned.
   */
  public function getOrCreateValidationError(?string $elementId): WebformSubmissionValidationError {
    $elementId = $this->normalizeElementId($elementId);
    return $this->validationErrors[$elementId] ?? new WebformSubmissionValidationError([], $elementId);
  }

  /**
   * Returns whether a validation error exists for the given element.
   *
   * @param string|null $elementId
   *   The ID of the Webform element to check for a validation error, or NULL to
   *   check for a general validation error.
   *
   * @return bool
   *   TRUE if a validation error exists for the given element.
   */
  public function hasValidationError(?string $elementId): bool {
    $elementId = $this->normalizeElementId($elementId);
    return isset($this->validationErrors[$elementId]);
  }

  /**
   * Adds a validation error.
   *
   * @param string $errorMessage
   *   The error message to add.
   * @param string|null $elementId
   *   The ID of the Webform element the error refers to, or NULL if the error
   *   is not element-specific.
   */
  public function addValidationError(string $errorMessage, ?string $elementId): static {
    $elementId = $this->normalizeElementId($elementId);
    $validationError = $this->getOrCreateValidationError($elementId);
    $validationError->addMessage($errorMessage);
    $this->validationErrors[$elementId] = $validationError;

    return $this;
  }

  /**
   * Returns whether the submission is valid.
   *
   * @return bool
   *   TRUE if there are no errors or validation errors.
   */
  public function isValid(): bool {
    return empty($this->errors) && empty($this->validationErrors);
  }

  /**
   * Transforms the element ID to the format used by the GraphQL client.
   *
   * Nested elements are represented in the Drupal Form API by an array of
   * parent elements and stored in the '#parents' property of the element. When
   * this is serialized to a string, the array keys are joined by '][' (example:
   * 'parent][child][grandchild'. This is different from the format used by the
   * GraphQL client, which uses a format similar to associative array paths
   * (example: 'parent[child][grandchild]').
   *
   * This method normalizes the element ID to the format used by the GraphQL
   * client.
   *
   * @param string|null $elementId
   *   The element ID to normalize.
   *
   * @return string|null
   *   The normalized element ID.
   *
   * @see \Drupal\Core\Form\FormElementHelper::getElementByName()
   */
  protected static function normalizeElementId(?string $elementId): ?string {
    if ($elementId === NULL) {
      return NULL;
    }

    // Check if the element ID needs to be normalized. It is only needed if it
    // contains a '][' sequence without a preceding '[' character.
    if (preg_match('/^[^\[]+]\[[^]]+/', $elementId)) {
      // Remove the first ']' character and append it to the end of the string.
      return preg_replace('/]/', '', $elementId, 1) . ']';
    }

    return $elementId;
  }

}
