<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Render\RendererInterface;
use Drupal\graphql\GraphQL\Execution\FieldContext;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\WebformInterface;
use Drupal\webform\WebformTokenManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Resolves the minimum or maximum date from a webform date element.
 *
 * The Webform module allows the use of relative dates such as "-1 week" and
 * tokens. This data producer resolves these to absolute dates in "YYYY-MM-DD"
 * format so they can be readily inserted into HTML date input fields in a
 * decoupled frontend.
 *
 * @DataProducer(
 *   id = "webform_date_extremum",
 *   name = @Translation("Webform date extremum"),
 *   description = @Translation("Returns the given minimum or maximum date in YYYY-MM-DD format."),
 *   produces = @ContextDefinition("string",
 *     label = @Translation("The date in YYYY-MM-DD format."),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "element" = @ContextDefinition("any",
 *       label = @Translation("Webform date based element")
 *     ),
 *     "extremum" = @ContextDefinition("string",
 *       label = @Translation("The extremum"),
 *       description = @Translation("Either 'min' or 'max' to indicate which extremum to resolve."),
 *     ),
 *     "webform" = @ContextDefinition("entity:webform",
 *       label = @Translation("Webform"),
 *       description = @Translation("The webform containing the date element."),
 *       required = TRUE
 *     )
 *   }
 * )
 */
class WebformDateExtremum extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected readonly RendererInterface $renderer,
    protected readonly WebformTokenManagerInterface $tokenManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('renderer'),
      $container->get('webform.token_manager'),
    );
  }

  /**
   * Resolves the given date extremum to an absolute date.
   *
   * @param array $element
   *   The webform date based element.
   * @param string $extremum
   *   The extremum to resolve, either "min" or "max".
   * @param \Drupal\webform\WebformInterface $webform
   *   The webform containing the date element.
   * @param \Drupal\graphql\GraphQL\Execution\FieldContext $context
   *   The field context.
   *
   * @return string|null
   *   The date in "YYYY-MM-DD" format or NULL if no value has been resolved.
   */
  public function resolve(array $element, string $extremum, WebformInterface $webform, FieldContext $context): ?string {
    assert(in_array($extremum, ['min', 'max'], TRUE));

    // In order to support tokens, we need to inspect the original element as it
    // is defined in the webform. The webform module actively removes token
    // based min/max values from the element so we cannot access them there.
    // @see \Drupal\webform\Plugin\WebformElement\DateBase::prepare()
    $elementKey = $element['#webform_key'];
    $property = '#date_date_' . $extremum;
    $originalValue = $webform->getElementDecoded($elementKey)[$property] ?? NULL;

    // If no value has been configured, we are done.
    if (empty($originalValue)) {
      return NULL;
    }

    // If the value contains the specific token indicating the creation date of
    // a webform submission, convert it to today's date. We don't have the
    // submission entity yet.
    if ($originalValue === '[webform_submission:created:html_date]') {
      $value = 'today';
    }

    // If the value contains any other tokens, render them now, capturing any
    // cacheability metadata they provide.
    elseif (str_contains($originalValue, '[')) {
      $renderContext = new RenderContext();
      $value = $this->renderer->executeInRenderContext($renderContext, function () use ($originalValue, $webform, $context) {
        $metadata = new BubbleableMetadata();
        $value = $this->tokenManager->replace($originalValue, $webform, [], [], $metadata);
        $context->addCacheableDependency($metadata);

        return $value;
      });
    }

    // If there are no tokens, we can use the original value as is.
    else {
      $value = $originalValue;
    }

    // If the value is empty, return NULL.
    if (empty($value)) {
      return NULL;
    }

    // Now that we have a concrete value, try to parse it as a date.
    $time = strtotime($value);
    if ($time === FALSE) {
      return NULL;
    }

    // Format the date as "YYYY-MM-DD" and return it.
    return date('Y-m-d', $time);
  }

}
