<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\Plugin\WebformElementInterface;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns details regarding multiple value support for a Webform element.
 *
 * @DataProducer(
 *   id = "webform_element_multiple_values",
 *   name = @Translation("Webform element multiple values"),
 *   description = @Translation("Returns information about multiple value support for a Webform element."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Any value"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "element" = @ContextDefinition(
 *       "any",
 *       label = @Translation("Webform element")
 *     )
 *   }
 * )
 */
class WebformElementMultipleValues extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Maps Webform element property names to GraphQL field names.
   */
  protected const PROPERTY_MAPPING = [
    'multiple_error' => 'message',
    'multiple__header_label' => 'headerLabel',
    'multiple__min_items' => 'minItems',
    'multiple__empty_items' => 'emptyItems',
    'multiple__add_more' => 'addMore',
    'multiple__add_more_items' => 'addMoreItems',
    'multiple__add_more_button_label' => 'addMoreButtonLabel',
    'multiple__add_more_input' => 'addMoreInput',
    'multiple__add_more_input_label' => 'addMoreInputLabel',
    'multiple__item_label' => 'itemLabel',
    'multiple__sorting' => 'sorting',
    'multiple__operations' => 'operations',
    'multiple__add' => 'add',
    'multiple__remove' => 'remove',
  ];

  /**
   * Constructs a new WebformElementMultipleValues instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $pluginId
   *   The plugin ID.
   * @param mixed $pluginDefinition
   *   The plugin definition.
   * @param \Drupal\webform\Plugin\WebformElementManagerInterface $elementManager
   *   The Webform element plugin manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected readonly WebformElementManagerInterface $elementManager,
    protected readonly RendererInterface $renderer,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.webform.element'),
      $container->get('renderer'),
    );
  }

  /**
   * Resolves multiple values information for a Webform element.
   *
   * @param array $element
   *   The Webform element for which to return the information.
   *
   * @return array|null
   *   The information about multiple values support, or NULL if the element
   *   does not support multiple values.
   */
  public function resolve(array $element): ?array {
    $plugin = $this->elementManager->getElementInstance($element);
    assert($plugin instanceof WebformElementInterface);
    // Contrary to the documentation, this method returns either a boolean or an
    // integer.
    $limit = $plugin->hasMultipleValues($element);
    if ($limit === FALSE) {
      return NULL;
    }

    $resolved = [];
    if ($limit) {
      // Convert an unlimited limit to -1 so it adheres to the GraphQL schema.
      $resolved['limit'] = $limit === TRUE ? -1 : $limit;
      // The no items message can be a render array. Convert it to a string.
      $resolved['noItemsMessage'] = $plugin->getElementProperty($element, 'multiple__no_items_message');
      if (!empty($resolved['noItemsMessage'])) {
        $resolved['noItemsMessage'] = (string) match (is_array($resolved['noItemsMessage'])) {
          TRUE => $this->renderer->renderInIsolation($resolved['noItemsMessage']),
          default => $resolved['noItemsMessage'],
        };
      }
      foreach (static::PROPERTY_MAPPING as $property => $key) {
        $resolved[$key] = $plugin->getElementProperty($element, $property);
      }
    }
    return $resolved;
  }

}
