<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\String\UnicodeString;

/**
 * Returns a single property from a Webform element.
 *
 * @DataProducer(
 *   id = "webform_element_property",
 *   name = @Translation("Webform element property"),
 *   description = @Translation("Returns the property of a Webform element."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Any value"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "element" = @ContextDefinition("any",
 *       label = @Translation("Webform element")
 *     ),
 *     "property" = @ContextDefinition("string",
 *       label = @Translation("Property")
 *     ),
 *     "type" = @ContextDefinition("string",
 *       label = @Translation("The desired type."),
 *       required = FALSE
 *     )
 *   }
 * )
 */
class WebformElementProperty extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new WebformElementProperty instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $pluginId
   *   The plugin ID.
   * @param mixed $pluginDefinition
   *   The plugin definition.
   * @param \Drupal\webform\Plugin\WebformElementManagerInterface $elementManager
   *   The Webform element plugin manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected readonly WebformElementManagerInterface $elementManager,
    protected readonly RendererInterface $renderer,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.webform.element'),
      $container->get('renderer'),
    );
  }

  /**
   * Resolves a property from a Webform element.
   *
   * @param array $element
   *   The Webform element from which to extract the property.
   * @param string $property
   *   The property to extract.
   * @param string $type
   *   The desired type of the property.
   *
   * @return mixed
   *   The value of the property.
   */
  public function resolve(array $element, string $property, string $type): mixed {
    // Derive the property name from the GraphQL field name, by converting it
    // from PascalCase to snake_case.
    $property = (new UnicodeString($property))->snake()->toString();

    // If this is an element with multiple values, inspect the first actual
    // element instead of the wrapper which is a fieldset or container.
    if ($element['#type'] === 'webform_multiple' && !empty($element['items'][0]['_item_'])) {
      $element = $element['items'][0]['_item_'];
    }

    $plugin = $this->elementManager->getElementInstance($element);
    $value = $plugin->getElementProperty($element, $property);

    if ($value === NULL) {
      return NULL;
    }

    return match ($type) {
      'string' => is_array($value) ? (string) $this->renderer->renderInIsolation($value) : (string) $value,
      'int' => (int) $value,
      'array' => is_array($value) ? $value : [],
      default => (bool) $value,
    };
  }

}
