<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\Utility\WebformArrayHelper;

/**
 * Returns the webform element states.
 *
 * @DataProducer(
 *   id = "webform_element_state",
 *   name = @Translation("Webform element states"),
 *   description = @Translation("Generate the webform element states."),
 *   produces = @ContextDefinition("map",
 *     label = @Translation("Webform element states"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "element" = @ContextDefinition("map",
 *       label = @Translation("Webform element")
 *     ),
 *     "state" = @ContextDefinition("string",
 *       label = @Translation("The state key (e.g. visible, required)")
 *     )
 *   }
 * )
 */
class WebformElementState extends DataProducerPluginBase {

  /**
   * Resolves the data related to the state of a webform element.
   *
   * @param array $element
   *   The webform element.
   * @param string $state
   *   The state.
   *
   * @return array|null
   *   The state configuration.
   */
  public function resolve(array $element, string $state) {
    // The state configuration.
    $config = $element[$state] ?? NULL;

    if (!$config) {
      return NULL;
    }

    $logic = $this->getConditionLogic($config);
    $conditions = $this->getConditions($config);

    if (empty($conditions)) {
      return NULL;
    }

    return [
      'logic' => $logic,
      'conditions' => $conditions,
    ];
  }

  /**
   * Determines the condition logic.
   *
   * @param array $conditions
   *   The webform conditions array.
   *
   * @return string
   *   The logic.
   */
  protected function getConditionLogic(array $conditions): string {
    if ($conditions && WebformArrayHelper::isSequential($conditions)) {
      if (in_array('xor', $conditions)) {
        return 'XOR';
      }
      elseif (in_array('or', $conditions)) {
        return 'OR';
      }
    }

    return 'AND';
  }

  /**
   * Maps the conditions.
   *
   * The conditions can either be an array with selectors as keys or a
   * sequential array that contains both conditions and strings indicating the
   * logic ('or', 'and', 'xor').
   *
   * @param array $definition
   *   The conditions array from the webform element.
   *
   * @return array
   *   The mapped, sequential array of conditions.
   */
  protected function getConditions(array $definition): array {
    if (!$definition) {
      return [];
    }

    $conditions = [];

    foreach ($definition as $index => $value) {
      // Skip and, or, and xor.
      if (is_string($value) && in_array($value, ['and', 'or', 'xor'])) {
        continue;
      }

      $selector = NULL;
      $condition = NULL;

      if (is_int($index)) {
        $selector = key($value);
        $condition = $value[$selector];
      }
      else {
        $selector = $index;
        $condition = $value;
      }

      $conditions[] = [
        'selector' => $selector,
        'trigger' => array_key_first($condition),
        'value' => (string) array_values($condition)[0],
      ];
    }

    return $conditions;
  }

}
