<?php

declare(strict_types=1);

namespace Drupal\graphql_webform\Plugin\GraphQL\DataProducer;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\graphql\Plugin\GraphQL\DataProducer\DataProducerPluginBase;
use Drupal\webform\Plugin\WebformElementManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns the regular expression pattern used to validate the element.
 *
 * @DataProducer(
 *   id = "webform_element_validation_pattern",
 *   name = @Translation("Webform element validation pattern"),
 *   description = @Translation("Returns the regular expression pattern used to validate the element."),
 *   produces = @ContextDefinition("any",
 *     label = @Translation("Any value"),
 *     required = FALSE
 *   ),
 *   consumes = {
 *     "element" = @ContextDefinition("any",
 *       label = @Translation("Webform element")
 *     )
 *   }
 * )
 */
class WebformElementValidationPattern extends DataProducerPluginBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new WebformElementValidationPattern instance.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $pluginId
   *   The plugin ID.
   * @param mixed $pluginDefinition
   *   The plugin definition.
   * @param \Drupal\webform\Plugin\WebformElementManagerInterface $elementManager
   *   The Webform element plugin manager.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    protected readonly WebformElementManagerInterface $elementManager,
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.webform.element'),
    );
  }

  /**
   * Resolves the validation pattern of a Webform element.
   *
   * @param array $element
   *   The Webform element from which to extract the validation pattern.
   *
   * @return mixed
   *   The validation pattern and its accompanying error message.
   */
  public function resolve(array $element): mixed {
    // If this is an element with multiple values, inspect the first actual
    // element instead of the wrapper which is a fieldset or container.
    if ($element['#type'] === 'webform_multiple' && !empty($element['items'][0]['_item_'])) {
      $element = $element['items'][0]['_item_'];
    }

    $plugin = $this->elementManager->getElementInstance($element);
    return [
      'rule' => $plugin->getElementProperty($element, 'pattern'),
      'message' => $plugin->getElementProperty($element, 'pattern_error'),
    ];
  }

}
